package zephyr

import (
	"errors"
	"strconv"
	"sync"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestConsolidatePlatforms(t *testing.T) {
	tests := []struct {
		Input    map[string]int64
		Expected map[string]int64
	}{
		{
			Input: map[string]int64{
				"console": 0,
				"tv":      3,
			},
			Expected: map[string]int64{
				"console": 3,
			},
		},
		{
			Input: map[string]int64{
				"desktop_app": 1000,
				"web":         1,
			},
			Expected: map[string]int64{
				"web": 1001,
			},
		},
	}

	for _, test := range tests {
		merged := consolidatePlatforms(test.Input)
		assert.Equal(t, test.Expected, merged, "certain platforms should be combined")
		assert.Equal(t, test.Input, merged, "passed-in map should be modified")
	}
}

func TestFetchOnceStrSuccess(t *testing.T) {
	fn := fetchOnceStr(func() (string, error) {
		return "value", nil
	})

	v, err := fn()
	require.Nil(t, err, "Error is nil")
	assert.Equal(t, v, "value", "returned value")
}

func TestFetchOnceStrFailure(t *testing.T) {
	fn := fetchOnceStr(func() (string, error) {
		return "", errors.New("error")
	})

	_, err := fn()
	require.Error(t, err, "returned Error")
}

func TestFetchOnceStrOnlyCallsOnce(t *testing.T) {
	var count int64
	fn := fetchOnceStr(func() (string, error) {
		time.Sleep(5 * time.Millisecond)
		count++
		return strconv.FormatInt(count, 10), nil
	})

	_, _ = fn()
	_, _ = fn()
	_, _ = fn()
	v, _ := fn()
	require.Equal(t, v, "1", "returned value")
}

func TestFetchOnceStrThreadSafe(t *testing.T) {
	var count int64
	fn := fetchOnceStr(func() (string, error) {
		time.Sleep(5 * time.Millisecond)
		count++
		return strconv.FormatInt(count, 10), nil
	})

	var wg sync.WaitGroup
	for i := 0; i < 5; i++ {
		go func() {
			_, _ = fn()
			wg.Done()
		}()
		wg.Add(1)
	}
	v, _ := fn()
	wg.Wait()
	require.Equal(t, v, "1", "returned value")
}
