package httputil

import (
	"context"
	"net/http"
	"time"
)

// Timeout sets a timeout on the http.Request.Context() for a given inner http.Handler.
//
// As with any context that reaches its deadline, the cancellation only works
// if the downstream context consumer leverages the cancellation functionality.
//
// For example:
// 1. Timeout set at 100ms
// 2. Inner handler's database query requires 99ms
// 3. JSON-serialization of the response body requires 2ms
// 4. Even though the context deadline is exceeded,
//    there is no consequence as the response writer does not consume the request context.
func Timeout(timeout time.Duration) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		middleware := func(w http.ResponseWriter, req *http.Request) {
			ctx, cancel := context.WithTimeout(req.Context(), timeout)
			defer cancel()

			inner.ServeHTTP(w, req.WithContext(ctx))
		}

		return http.HandlerFunc(middleware)
	}
}
