package statsd

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/commerce/splatter"
	"code.justin.tv/common/gometrics"
	"github.com/cactus/go-statsd-client/statsd"
	log "github.com/sirupsen/logrus"
)

const (
	flushInterval        = 10 * time.Second
	gometricsMonitorRate = 1 * time.Second
	sampleRate           = 1.0
)

// Client wraps the statsd StatSender
type Client struct {
	Statter statsd.Statter
}

// NewClient instantiates a new Client
func NewClient(stage string) Client {
	buffConfig := &splatter.BufferedTelemetryConfig{
		FlushPeriod:       flushInterval,
		BufferSize:        100000,
		AggregationPeriod: time.Minute,
		ServiceName:       "Dashy",
		AWSRegion:         "us-west-2",
		Stage:             stage,
		Substage:          "live",
	}
	statter := splatter.NewBufferedTelemetryCloudWatchStatter(buffConfig, nil)
	log.WithFields(log.Fields{"Stage": stage}).Info("Set up statter client")

	gometrics.Monitor(statter, gometricsMonitorRate)

	return Client{Statter: statter}
}

// MetricCollector is a middleware that wraps a given http.Handler,
// collecting and sending metrics to StatsD.
func (c *Client) MetricCollector(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		loggedRW := &loggedResponseWriter{
			ResponseWriter: w,
		}

		start := time.Now()

		inner.ServeHTTP(loggedRW, r)

		duration := time.Since(start)
		path := normalizePath(r.URL.EscapedPath())
		name := fmt.Sprintf("request.%s.%s.%d", r.Method, path, loggedRW.status)

		go func() {
			if err := c.Statter.TimingDuration(name, duration, sampleRate); err != nil {
				log.WithError(err).WithFields(log.Fields{
					"stat_name": name,
					"duration":  duration,
				}).Warn("statsd: failed to send timing duration")
			}

			if err := c.Statter.Inc(name, 1, sampleRate); err != nil {
				log.WithError(err).WithField("stat_name", name).Warn("statsd: failed to send increment stat")
			}
		}()
	}

	return http.HandlerFunc(middleware)
}

type loggedResponseWriter struct {
	status int
	http.ResponseWriter
}

// WriteHeader allows loggedResponseWriter to implement http.ResponseWriter
// but keeps track of the status code.
func (w *loggedResponseWriter) WriteHeader(status int) {
	w.status = status
	w.ResponseWriter.WriteHeader(status)
}

func normalizePath(path string) string {
	var keptSegments []string
	segments := strings.Split(strings.Trim(path, "/"), "/")

	// Strip any numbers that might be IDs out of the stat name,
	// since Graphite cannot handle them.
	for _, segment := range segments {
		if _, err := strconv.Atoi(segment); err != nil {
			keptSegments = append(keptSegments, segment)
		}
	}

	return strings.Join(keptSegments, "_")
}
