resource "aws_vpc" "main" {
  cidr_block           = "${var.reserved_vpc_cidr}"
  instance_tenancy     = "default"
  enable_dns_support   = "true"
  enable_dns_hostnames = "true"

  tags {
    Name = "MainVPC"
  }
}

resource "aws_internet_gateway" "main" {
  vpc_id = "${aws_vpc.main.id}"

  tags {
    Name = "MainInternetGateway"
  }
}

locals {
  vpc_id = "${aws_vpc.main.id}"
  public_subnet_shift = 4
  public_subnet_offset = 12
  private_subnet_shift = 2
}

# 3 Public Subnets

resource "aws_subnet" "public_a" {
  vpc_id                  = "${aws_vpc.main.id}"
  availability_zone       = "us-west-2a"
  cidr_block              = "${cidrsubnet(aws_vpc.main.cidr_block, local.public_subnet_shift, local.public_subnet_offset)}"
  map_public_ip_on_launch = true

  tags {
    Name = "Public A"
  }
}

resource "aws_subnet" "public_b" {
  vpc_id                  = "${aws_vpc.main.id}"
  availability_zone       = "us-west-2b"
  cidr_block              = "${cidrsubnet(aws_vpc.main.cidr_block, local.public_subnet_shift, local.public_subnet_offset + 1)}"
  map_public_ip_on_launch = true

  tags {
    Name = "Public B"
  }
}

resource "aws_subnet" "public_c" {
  vpc_id                  = "${aws_vpc.main.id}"
  availability_zone       = "us-west-2c"
  cidr_block              = "${cidrsubnet(aws_vpc.main.cidr_block, local.public_subnet_shift, local.public_subnet_offset + 2)}"
  map_public_ip_on_launch = true

  tags {
    Name = "Public C"
  }
}

# 3 Private Subnets

resource "aws_subnet" "private_a" {
  vpc_id            = "${aws_vpc.main.id}"
  availability_zone = "us-west-2a"
  cidr_block        = "${cidrsubnet(aws_vpc.main.cidr_block, local.private_subnet_shift, 0)}"

  tags {
    Name = "Private A"
  }
}

resource "aws_subnet" "private_b" {
  vpc_id            = "${aws_vpc.main.id}"
  availability_zone = "us-west-2b"
  cidr_block        = "${cidrsubnet(aws_vpc.main.cidr_block, local.private_subnet_shift, 1)}"

  tags {
    Name = "Private B"
  }
}

resource "aws_subnet" "private_c" {
  vpc_id            = "${aws_vpc.main.id}"
  availability_zone = "us-west-2c"
  cidr_block        = "${cidrsubnet(aws_vpc.main.cidr_block, local.private_subnet_shift, 2)}"

  tags {
    Name = "Private C"
  }
}

locals {
  private_subnets = ["${aws_subnet.private_a.id}", "${aws_subnet.private_b.id}", "${aws_subnet.private_c.id}"]
}

# Route Tables

resource "aws_route_table" "external" {
  vpc_id = "${aws_vpc.main.id}"

  tags {
    Name = "External"
  }
}

resource "aws_route_table" "private_a" {
  vpc_id = "${aws_vpc.main.id}"

  tags {
    Name = "Private A"
  }
}

resource "aws_route_table" "private_b" {
  vpc_id = "${aws_vpc.main.id}"

  tags {
    Name = "Private B"
  }
}

resource "aws_route_table" "private_c" {
  vpc_id = "${aws_vpc.main.id}"

  tags {
    Name = "Private C"
  }
}

# Route Table Associations

## Public

resource "aws_route_table_association" "public_a" {
  subnet_id      = "${aws_subnet.public_a.id}"
  route_table_id = "${aws_route_table.external.id}"
}

resource "aws_route_table_association" "public_b" {
  subnet_id      = "${aws_subnet.public_b.id}"
  route_table_id = "${aws_route_table.external.id}"
}

resource "aws_route_table_association" "public_c" {
  subnet_id      = "${aws_subnet.public_c.id}"
  route_table_id = "${aws_route_table.external.id}"
}

## Private

resource "aws_route_table_association" "private_a" {
  subnet_id      = "${aws_subnet.private_a.id}"
  route_table_id = "${aws_route_table.private_a.id}"
  depends_on     = ["aws_subnet.private_a"]
}

resource "aws_route_table_association" "private_b" {
  subnet_id      = "${aws_subnet.private_b.id}"
  route_table_id = "${aws_route_table.private_b.id}"
  depends_on     = ["aws_subnet.private_a"]
}

resource "aws_route_table_association" "private_c" {
  subnet_id      = "${aws_subnet.private_c.id}"
  route_table_id = "${aws_route_table.private_c.id}"
  depends_on     = ["aws_subnet.private_c"]
}

# NAT for public subnets

resource "aws_eip" "a" {
  vpc = true
}

resource "aws_eip" "b" {
  vpc = true
}

resource "aws_eip" "c" {
  vpc = true
}

resource "aws_nat_gateway" "gw_a" {
  allocation_id = "${aws_eip.a.id}"
  subnet_id     = "${aws_subnet.public_a.id}"
}

resource "aws_nat_gateway" "gw_b" {
  allocation_id = "${aws_eip.b.id}"
  subnet_id     = "${aws_subnet.public_b.id}"
}

resource "aws_nat_gateway" "gw_c" {
  allocation_id = "${aws_eip.c.id}"
  subnet_id     = "${aws_subnet.public_c.id}"
}

# Routes

resource "aws_route" "default_public" {
  route_table_id         = "${aws_route_table.external.id}"
  destination_cidr_block = "0.0.0.0/0"
  gateway_id             = "${aws_internet_gateway.main.id}"
  depends_on             = ["aws_route_table.external"]
}

resource "aws_route" "default_private_a" {
  route_table_id         = "${aws_route_table.private_a.id}"
  nat_gateway_id         = "${aws_nat_gateway.gw_a.id}"
  destination_cidr_block = "0.0.0.0/0"
  depends_on             = ["aws_route_table.private_a", "aws_nat_gateway.gw_a"]
}

resource "aws_route" "default_private_b" {
  route_table_id         = "${aws_route_table.private_b.id}"
  nat_gateway_id         = "${aws_nat_gateway.gw_b.id}"
  destination_cidr_block = "0.0.0.0/0"
  depends_on             = ["aws_route_table.private_b", "aws_nat_gateway.gw_b"]
}

resource "aws_route" "default_private_c" {
  route_table_id         = "${aws_route_table.private_c.id}"
  nat_gateway_id         = "${aws_nat_gateway.gw_c.id}"
  destination_cidr_block = "0.0.0.0/0"
  depends_on             = ["aws_route_table.private_c", "aws_nat_gateway.gw_c"]
}
