resource "aws_cloudwatch_metric_alarm" "average_latency" {
  alarm_name                = "${var.elastic_beanstalk_environment_name}-average-latency"
  alarm_description         = "Average ELB latency for ${var.elastic_beanstalk_environment_name}"
  namespace                 = "AWS/ElasticBeanstalk"
  metric_name               = "ApplicationLatencyP90"
  statistic                 = "Average"
  comparison_operator       = "GreaterThanThreshold"
  threshold                 = "0.5"                                                               # seconds
  period                    = "60"                                                                # seconds
  evaluation_periods        = "10"
  datapoints_to_alarm       = "10"
  alarm_actions             = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  ok_actions                = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  treat_missing_data        = "notBreaching"

  dimensions {
    EnvironmentName = "${var.elastic_beanstalk_environment_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "environment_health" {
  alarm_name                = "${var.elastic_beanstalk_environment_name}-health"
  alarm_description         = "Monitors the health of this EBS application"
  namespace                 = "AWS/ElasticBeanstalk"
  metric_name               = "EnvironmentHealth"
  statistic                 = "Maximum"
  comparison_operator       = "GreaterThanOrEqualToThreshold"
  threshold                 = "20"                                               # 20 is degraded health
  period                    = "60"
  evaluation_periods        = "4"
  alarm_actions             = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  ok_actions                = ["${aws_sns_topic.pagerduty_alerts.arn}"]

  dimensions {
    EnvironmentName = "${var.elastic_beanstalk_environment_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "disk_usage" {
  alarm_name                = "${var.elastic_beanstalk_environment_name}-disk_usage"
  alarm_description         = "Monitors disk usage of elastic beanstalk instance"
  namespace                 = "AWS/ElasticBeanstalk"
  metric_name               = "RootFilesystemUtil"
  statistic                 = "Average"
  comparison_operator       = "GreaterThanOrEqualToThreshold"
  threshold                 = "80"                                  
  period                    = "60"
  evaluation_periods        = "20"
  alarm_actions             = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  ok_actions                = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  treat_missing_data        = "notBreaching"

  dimensions {
    EnvironmentName = "${var.elastic_beanstalk_environment_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "backend_5XX" {
  alarm_name          = "${var.elastic_beanstalk_environment_name}-backend-5XX"
  alarm_description   = "Monitors the number of backend 5XX responses"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = 5
  threshold           = 10
  treat_missing_data  = "notBreaching"

  alarm_actions             = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  ok_actions                = ["${aws_sns_topic.pagerduty_alerts.arn}"]

  metric_query {
    id = "percentage_5xx"
    expression = "error_req/total_req*100"
    label = "Percentage 5xx Errors"
    return_data = "true"
  }

  metric_query {
    id = "error_req"
    metric {
      metric_name = "ApplicationRequests5xx"
      namespace = "AWS/ElasticBeanstalk"
      period = 60
      stat = "Sum"
      unit = "Count"
      dimensions {
        EnvironmentName = "${var.elastic_beanstalk_environment_name}"
      }
    }
  }

  metric_query {
    id = "total_req"
    metric {
      metric_name = "ApplicationRequestsTotal"
      namespace = "AWS/ElasticBeanstalk"
      period = 60
      stat = "Sum"
      unit = "Count"
      dimensions {
        EnvironmentName = "${var.elastic_beanstalk_environment_name}"
      }
    }
  }
}

resource "aws_cloudwatch_metric_alarm" "low_request_count" {
  alarm_name          = "${var.elastic_beanstalk_environment_name}-low_request_count"
  alarm_description   = "Total application requests under threshold"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = 5
  metric_name         = "ApplicationRequestsTotal"
  namespace           = "AWS/ElasticBeanstalk"
  period              = 60
  statistic           = "Sum"
  threshold           = 20
  treat_missing_data  = "breaching"
  dimensions {
    EnvironmentName = "${var.elastic_beanstalk_environment_name}"
  }

  alarm_actions             = ["${aws_sns_topic.pagerduty_alerts.arn}"]
  ok_actions                = ["${aws_sns_topic.pagerduty_alerts.arn}"]
}
