package clips

import (
	"errors"
	"strconv"
	"time"
)

const (
	ErrorEndTimeBeforeStartTime = "StartTime must be before EndTime"
	ErrorInvalidChannelID       = "Invalid channel id"
	ErrorChannelIDNotParsable   = "Channel id not parsable"
)

// GetV1ClipsByTimeReqParams contains the REQUIRED parameters for making a
// GET request for a channel's clips activity within StartTime and EndTime
type GetV1ClipsByTimeReqParams struct {
	ChannelID string
	StartTime time.Time
	EndTime   time.Time
}

// Meta contains the metadata of the Clips response
//
// While StartTime and EndTime are the filters,
// it is possible for either or both to be null.
//
// SessionGapDurationMinutes is the preconfigured duration
// of the session gap in minutes. When a broadcast ends
// and a subsequent broadcast starts within the configured
// gap duration, the two broadcasts are considered one session.
type Meta struct {
	StartTime                 *time.Time `json:"start_time"`
	EndTime                   *time.Time `json:"end_time"`
	SessionGapDurationMinutes int        `json:"session_gap_duration_minutes"`
}

type Response struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
	Meta    Meta   `json:"meta"`
	Data    Data   `json:"data"`
}

type Data struct {
	Clips []*Clips `json:"clips"`
}

// Clips is a time series snapshot of clips data at Timestamp.
type Clips struct {
	Timestamp      *time.Time       `json:"timestamp"`
	Plays          int64            `json:"plays"`
	Creates        int64            `json:"creates"`
	PlaysBreakdown map[string]int64 `json:"plays_breakdown"`
}

func (params *GetV1ClipsByTimeReqParams) Validate() error {
	if params.EndTime.Before(params.StartTime) {
		return errors.New(ErrorEndTimeBeforeStartTime)
	}

	channelIDInt, err := strconv.Atoi(params.ChannelID)
	if err != nil {
		return errors.New(ErrorChannelIDNotParsable)
	}
	if channelIDInt < 1 {
		return errors.New(ErrorInvalidChannelID)
	}

	return nil
}
