package followers

import (
	"errors"
	"strconv"
	"time"
)

const (
	ErrorEndTimeBeforeStartTime = "StartTime must be before EndTime"
	ErrorInvalidChannelID       = "Invalid channel id"
	ErrorChannelIDNotParsable   = "Channel id not parsable"
)

// GetV1FollowersByTimeReqParams contains the REQUIRED parameters for making a
// GET request for a channel's follower activity within StartTime and EndTime
type GetV1FollowersByTimeReqParams struct {
	ChannelID string
	StartTime time.Time
	EndTime   time.Time
}

// Response is the base payload signature of the followers API
type Response struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
	Meta    Meta   `json:"meta"`
	Data    Data   `json:"data"`
}

// Meta contains the metadata of the followers response
//
// While StartTime and EndTime are the filters,
// it is possible for either or both to be null.
//
// SessionGapDurationMinutes is the preconfigured duration
// of the session gap in minutes. When a broadcast ends
// and a subsequent broadcast starts within the configured
// gap duration, the two broadcasts are considered one session.
type Meta struct {
	StartTime                 *time.Time `json:"start_time"`
	EndTime                   *time.Time `json:"end_time"`
	SessionGapDurationMinutes int        `json:"session_gap_duration_minutes"`
	InterpolationChunkMinutes int        `json:"interpolation_chunk_minutes"`
}

// Data contains the various time series for a channel's statistics
type Data struct {
	FollowersChange []*FollowersChangeCount `json:"followers_change"`
}

// FollowersChangeCount is a time series of follower activity
//
// Count is the number of followers gained during the broadcast at the
// respective timestamp
type FollowersChangeCount struct {
	Timestamp *time.Time `json:"timestamp"`
	Count     int64      `json:"count"`
}

func (params *GetV1FollowersByTimeReqParams) Validate() error {
	if params.EndTime.Before(params.StartTime) {
		return errors.New(ErrorEndTimeBeforeStartTime)
	}

	channelIDInt, err := strconv.Atoi(params.ChannelID)
	if err != nil {
		return errors.New(ErrorChannelIDNotParsable)
	}
	if channelIDInt < 1 {
		return errors.New(ErrorInvalidChannelID)
	}

	return nil
}
