package vpdemographics

import (
	"errors"
	"strconv"
	"time"
)

const (
	ErrorEndTimeBeforeStartTime = "StartTime must be before EndTime"
	ErrorInvalidChannelID       = "Invalid channel id"
	ErrorChannelIDNotParsable   = "Channel id not parsable"
)

// GetV1VideoPlayDemographicsByTimeReqParams contains the REQUIRED parameters for making a
// GET request for a channel's video play demographics within StartTime and EndTime
type GetV1VideoPlayDemographicsByTimeReqParams struct {
	ChannelID string
	StartTime time.Time
	EndTime   time.Time
}

// Response is the base payload signature of the vpdemographics API
type Response struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
	Meta    Meta   `json:"meta"`
	Data    Data   `json:"data"`
}

// Meta contains the metadata of the vpdemographics response
//
// While StartTime and EndTime are the filters,
// it is possible for either or both to be null.
//
// SessionGapDurationMinutes is the preconfigured duration
// of the session gap in minutes. When a broadcast ends
// and a subsequent broadcast starts within the configured
// gap duration, the two broadcasts are considered one session.
type Meta struct {
	StartTime                 *time.Time `json:"start_time"`
	EndTime                   *time.Time `json:"end_time"`
	SessionGapDurationMinutes int        `json:"session_gap_duration_minutes"`
	VideoPlayCount            int64      `json:"video_play_count"`
}

// Data contains the various time series for a channel's statistics
type Data struct {
	Demographics []*VideoPlayDemographics `json:"demographics"`
}

// VideoPlayDemographics contains a geographical, platform and referall
// breakdown for all video play events in the chunk
type VideoPlayDemographics struct {
	Timestamp    *time.Time        `json:"timestamp"`
	Geographical map[string]int64  `json:"geo"`
	Platform     map[string]int64  `json:"platform"`
	Referral     ReferralBreakdown `json:"referral"`
}

// ReferralBreakdown splits video play referral sources between Internal
// (from within twitch.tv domain) and External
type ReferralBreakdown struct {
	Internal        map[string]int64 `json:"internal"`
	InternalTwitch  map[string]int64 `json:"internal_twitch"`
	InternalChannel map[string]int64 `json:"internal_channel"`
	External        map[string]int64 `json:"external"`
}

func (params *GetV1VideoPlayDemographicsByTimeReqParams) Validate() error {
	if params.EndTime.Before(params.StartTime) {
		return errors.New(ErrorEndTimeBeforeStartTime)
	}

	channelIDInt, err := strconv.Atoi(params.ChannelID)
	if err != nil {
		return errors.New(ErrorChannelIDNotParsable)
	}
	if channelIDInt < 1 {
		return errors.New(ErrorInvalidChannelID)
	}

	return nil
}
