package main

import (
	"fmt"
	"net/http"
	"os"

	"code.justin.tv/cb/dropship/internal/clients/dynamodb"
	"code.justin.tv/cb/dropship/internal/clients/ripley"
	"code.justin.tv/cb/dropship/internal/clients/stats"
	"code.justin.tv/cb/dropship/internal/dropshipserver"
	"code.justin.tv/cb/dropship/internal/util"
	"code.justin.tv/cb/dropship/rpc/dropship"

	"github.com/twitchtv/twirp"
	"goji.io/pat"

	log "github.com/sirupsen/logrus"
	statsdhook "github.com/twitchtv/twirp/hooks/statsd"
	goji "goji.io"
)

func main() {
	port := os.Getenv("PORT")
	if port == "" {
		log.Fatal("dropship: no port found in env")
		return
	}

	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("dropship: no environment found in env")
		return
	}

	region := os.Getenv("REGION")
	if region == "" {
		log.Fatal("dropship: no region found in env")
		return
	}

	dynamoTableName := os.Getenv("LAYOUT_TABLE_NAME")
	if dynamoTableName == "" {
		log.Fatal("dropship: no dynamo table name found in env")
		return
	}

	dynamoCredentials := util.GetCredsForEC2(env, region)
	dynamoDBClient, err := dynamodb.NewClient(dynamoTableName, region, dynamoCredentials) // TODO: get table name in config
	if err != nil {
		log.Fatal("dropship: failed to create dynamodb client", err)
		return
	}

	statsHost := os.Getenv("STATS_HOST")
	if statsHost == "" {
		log.Fatal("dropship: no stats host found in env")
		return
	}
	statsClient, err := stats.NewClient(statsHost, env)
	if err != nil {
		log.Fatal("dropship: could not create statsd client: ", err)
		return
	}

	ripleyHost := os.Getenv("RIPLEY_HOST")
	if ripleyHost == "" {
		log.Fatal("dropship: no ripley host found in env")
		return
	}
	ripleyClient := ripley.NewClient(ripleyHost)

	server := &dropshipserver.Server{
		DynamoDB: dynamoDBClient,
		Stats:    statsClient,
		Ripley:   ripleyClient,
	}

	statsHook := statsdhook.NewStatsdServerHooks(statsClient.Statter)
	hooks := twirp.ChainHooks(statsHook)
	twirpHandler := dropship.NewDropshipServer(server, hooks)

	mux := goji.NewMux()
	mux.Handle(pat.Post(dropship.DropshipPathPrefix+"*"), twirpHandler)
	mux.HandleFunc(pat.Get("/health"), util.HealthCheck)

	addr := fmt.Sprintf(":%s", port)
	log.Infof("dropship: server listening on port %s", addr)

	if err := http.ListenAndServe(addr, mux); err != nil {
		log.Fatal("server error: ", err)
	}
}
