package dynamodb

import (
	"context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/pkg/errors"
)

// Database is the interface for the dynamodb client
type Database interface {
	GetLayout(ctx context.Context, ownerID, channelID string) (Layout, error)
	SetLayout(ctx context.Context, ownerID, channelID string, layout Layout) (string, error)
	AppendToLayout(ctx context.Context, ownerID, channelID, layoutID string, item Position) error
	DeleteLayout(ctx context.Context, ownerID, channelID string) error
}

// Client wraps the dynamodb API
type Client struct {
	layoutTableName string
	dynamoDB        dynamodbiface.DynamoDBAPI
}

// NewClient creates a new dynamo client
func NewClient(layoutTable string, region string, creds *credentials.Credentials) (*Client, error) {
	conf := &aws.Config{
		Credentials: creds,
		MaxRetries:  aws.Int(3),
		Region:      aws.String(region),
	}

	sess, err := session.NewSession(conf)
	if err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to initialize dynamo db client")
	}

	return &Client{
		layoutTableName: layoutTable,
		dynamoDB:        dynamodb.New(sess),
	}, nil
}
