package dynamodb

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/gofrs/uuid"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

// GetLayout returns an array of positions for quick action items given an owner id.
func (c *Client) GetLayout(ctx context.Context, ownerID, channelID string) (Layout, error) {
	input := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("#O = :ownerID AND #C = :channelID"),
		ExpressionAttributeNames: map[string]*string{
			"#O": aws.String("owner_id"),
			"#C": aws.String("channel_id"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":ownerID":   {S: aws.String(ownerID)},
			":channelID": {S: aws.String(channelID)},
		},
		ExclusiveStartKey: nil,
		ScanIndexForward:  aws.Bool(true),
		TableName:         aws.String(c.layoutTableName),
	}

	output, err := c.dynamoDB.QueryWithContext(ctx, input)
	if err != nil {
		return Layout{}, err
	}

	var items []Layout
	err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &items)
	if err != nil {
		return Layout{}, err
	}

	if len(items) == 0 {
		return Layout{
			OwnerID:   ownerID,
			ChannelID: channelID,
		}, nil
	}

	return items[0], nil
}

func (c *Client) SetLayout(ctx context.Context, ownerID, channelID string, layout Layout) (string, error) {
	listItems, err := dynamodbattribute.Marshal(layout.Items)
	if err != nil {
		return "", err
	}

	layoutID, err := uuid.NewV4()
	if err != nil {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"owner_id":   ownerID,
		}).WithError(err).Error("dynamodb: unable to generate UUID")
		return "", err
	}

	input := &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"owner_id":     {S: aws.String(ownerID)},
			"channel_id":   {S: aws.String(channelID)},
			"layout_id":    {S: aws.String(layoutID.String())},
			"layout":       listItems,
			"last_updated": {S: aws.String(time.Now().UTC().Format(time.RFC3339Nano))},
		},
		TableName: aws.String(c.layoutTableName),
	}

	_, err = c.dynamoDB.PutItemWithContext(ctx, input)
	return layoutID.String(), err
}

func (c *Client) AppendToLayout(ctx context.Context, ownerID, channelID string, layoutID string, item Position) error {
	newItem, err := dynamodbattribute.Marshal([]Position{item})
	if err != nil {
		return err
	}

	input := &dynamodb.UpdateItemInput{
		Key: map[string]*dynamodb.AttributeValue{
			"owner_id": {
				S: aws.String(ownerID),
			},
			"channel_id": {
				S: aws.String(channelID),
			},
		},
		ConditionExpression: aws.String("#O = :ownerID AND #C = :channelID AND #LI = :layoutID"),
		ExpressionAttributeNames: map[string]*string{
			"#O":  aws.String("owner_id"),
			"#C":  aws.String("channel_id"),
			"#L":  aws.String("layout"),
			"#LI": aws.String("layout_id"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":ownerID":   {S: aws.String(ownerID)},
			":channelID": {S: aws.String(channelID)},
			":layoutID":  {S: aws.String(layoutID)},
			":newItem":   newItem,
		},
		TableName:        aws.String(c.layoutTableName),
		UpdateExpression: aws.String("set layout = list_append(#L, :newItem)"),
	}

	_, err = c.dynamoDB.UpdateItemWithContext(ctx, input)

	return err
}

func (c *Client) DeleteLayout(ctx context.Context, ownerID, channelID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: aws.String(c.layoutTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id": {S: aws.String(channelID)},
			"owner_id":   {S: aws.String(ownerID)},
		},
	}

	_, err := c.dynamoDB.DeleteItemWithContext(ctx, input)
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to delete layout")
	}

	return nil
}
