package dropshipserver

import (
	"context"
	"errors"
	"strings"

	"code.justin.tv/cb/dropship/internal/clients/dynamodb"
	"code.justin.tv/cb/dropship/internal/clients/ripley"
	"code.justin.tv/cb/dropship/internal/clients/stats"
	"code.justin.tv/cb/dropship/internal/quickactions"

	pb "code.justin.tv/cb/dropship/rpc/dropship"
	log "github.com/sirupsen/logrus"
)

// Server implements the Dropship service.
type Server struct {
	DynamoDB dynamodb.Database
	Stats    stats.StatSender
	Ripley   ripley.Payouts
}

func (s *Server) handleDefaultLayout(ctx context.Context, ownerID, channelID string) (*pb.GetLayoutResp, error) {
	isPartner, err := s.Ripley.IsPartner(ctx, channelID)
	if err != nil {
		log.Error("dropship: Ripley.IsPartner failed to get Partner status")
		return nil, err
	}
	defaultLayout := quickactions.GetDefaultLayout(isPartner, ownerID, channelID)

	layout, err := s.SetLayout(ctx, defaultLayout)
	if err != nil {
		log.Error("dropship: SetLayout failed to set layout")
		return nil, err
	}
	return &pb.GetLayoutResp{
		OwnerID:   ownerID,
		ChannelID: channelID,
		LayoutID:  layout.LayoutID,
		Layout:    layout.Layout,
	}, nil
}

// GetLayout returns a list of quick action IDs indicating the order in which those actions should appear.
func (s *Server) GetLayout(ctx context.Context, req *pb.GetLayoutReq) (*pb.GetLayoutResp, error) {
	ownerID := req.GetOwnerID()
	if len(ownerID) == 0 {
		log.Error("dropship: UpdatePositionInLayout request contained no owner id")
		return nil, ErrMissingOwnerID
	}

	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Error("dropship: AddToLayout request contained no channel id")
		return nil, ErrMissingChannelID
	}

	layout, err := s.DynamoDB.GetLayout(ctx, ownerID, channelID)
	if err != nil {
		log.WithError(err).Error("dropship: GetLayout failed to get layout")
		return nil, err
	}

	items := convertDDBLayoutToItems(layout)

	// If we have no existing layout, we fill it with the default layout based on payout status.
	if len(items) == 0 {
		return s.handleDefaultLayout(ctx, ownerID, channelID)
	}

	return &pb.GetLayoutResp{
		OwnerID:   ownerID,
		ChannelID: channelID,
		LayoutID:  layout.LayoutID,
		Layout:    items,
	}, nil
}

func (s *Server) UpdateLayout(ctx context.Context, req *pb.UpdateLayoutReq) (*pb.UpdateLayoutResp, error) {
	ownerID := req.GetOwnerID()
	if len(ownerID) == 0 {
		log.Error("dropship: UpdateLayout request contained no owner id")
		return nil, ErrMissingOwnerID
	}

	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Error("dropship: UpdateLayout request contained no channel id")
		return nil, ErrMissingChannelID
	}

	item := req.GetItem()
	if item == nil {
		log.Error("dropship: UpdateLayout request contained no item")
		return nil, ErrMissingItem
	}

	layoutID := req.GetLayoutID()
	if len(layoutID) == 0 {
		log.Error("dropship: UpdateLayout request contained no layout ID")
		return nil, ErrMissingLayoutID
	}

	err := s.DynamoDB.AppendToLayout(ctx, ownerID, channelID, layoutID, convertItemToPosition(item))
	if err != nil {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"owner_id":   ownerID,
			"layout_id":  layoutID,
			"item":       item,
		}).WithError(err).Error("dropship: UpdateLayout failed to append item to layout")
		return nil, err
	}

	return &pb.UpdateLayoutResp{}, nil
}

func (s *Server) SetLayout(ctx context.Context, req *pb.SetLayoutReq) (*pb.SetLayoutResp, error) {
	ownerID := req.GetOwnerID()
	if len(ownerID) == 0 {
		log.Error("dropship: SetLayout request contained no owner id")
		return nil, ErrMissingOwnerID
	}

	channelID := req.GetChannelID()
	if len(channelID) == 0 {
		log.Error("dropship: SetLayout request contained no channel id")
		return nil, ErrMissingChannelID
	}

	layout := req.GetLayout()
	if len(layout) == 0 {
		return nil, errors.New("missing layout")
	}

	invalids, err := quickactions.ValidateLayout(layout, 0)
	if err != nil {
		return nil, err
	}

	if len(invalids.InvalidTypes) != 0 {
		return nil, GetErrInvalidQuickActions(invalids.InvalidTypes)
	}

	if len(invalids.InvalidNames) != 0 {
		return nil, GetErrInvalidQuickActions(invalids.InvalidNames)
	}

	items := convertLayoutToPositionList(layout)
	ddbLayout := dynamodb.Layout{
		OwnerID:   ownerID,
		ChannelID: channelID,
		Items:     items,
	}

	layoutID, err := s.DynamoDB.SetLayout(ctx, ownerID, channelID, ddbLayout)
	if err != nil {
		return nil, err
	}

	return &pb.SetLayoutResp{
		OwnerID:   ownerID,
		ChannelID: channelID,
		LayoutID:  layoutID,
		Layout:    layout,
	}, nil
}

func convertDDBLayoutToItems(layout dynamodb.Layout) []*pb.Item {
	result := make([]*pb.Item, len(layout.Items))
	for i, item := range layout.Items {
		curr := &pb.Item{
			Type: item.Type,
			IDs:  convertPositionListToItems(item.IDs),
		}
		if item.Name != nil {
			curr.Name = *item.Name
		}
		if item.ID != nil {
			curr.ID = *item.ID
		}
		result[i] = curr
	}
	return result
}

func convertPositionListToItems(positions []dynamodb.Position) []*pb.Item {
	if len(positions) == 0 { // also covers nil case
		return nil
	}

	items := make([]*pb.Item, len(positions))
	for i, pos := range positions {
		curr := &pb.Item{
			Type: pos.Type,
			IDs:  convertPositionListToItems(pos.IDs),
		}
		if pos.Name != nil {
			curr.Name = *pos.Name
		}
		if pos.ID != nil {
			curr.ID = *pos.ID
		}
		items[i] = curr
	}

	return items
}

func convertItemToPosition(item *pb.Item) dynamodb.Position {
	// We should always have a type, not matter what the rest of the input is.
	pos := dynamodb.Position{
		Type: item.Type,
	}

	// If we have a name, this is a folder.
	if len(item.Name) != 0 {
		pos.Name = &item.Name
		pos.IDs = convertLayoutToPositionList(item.IDs)
	}

	// If we have an ID, this is a single quick action.
	if len(item.ID) != 0 {
		newID := strings.ToLower(item.ID)
		pos.ID = &newID
	}
	return pos
}

func convertLayoutToPositionList(layout []*pb.Item) []dynamodb.Position {
	if len(layout) == 0 {
		return nil
	}

	items := make([]dynamodb.Position, len(layout))
	for idx := range layout {
		items[idx] = convertItemToPosition(layout[idx])
	}
	return items
}
