package dropshipserver

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/cb/dropship/rpc/dropship"
	"github.com/golang/mock/gomock"

	dynamomodels "code.justin.tv/cb/dropship/internal/clients/dynamodb"
	dynamodb "code.justin.tv/cb/dropship/internal/mocks/dynamodb"
	ripley "code.justin.tv/cb/dropship/internal/mocks/ripley"
	stats "code.justin.tv/cb/dropship/internal/mocks/stats"
	pb "code.justin.tv/cb/dropship/rpc/dropship"
	log "github.com/sirupsen/logrus"
)

const (
	channelID = "12345"
	ownerID   = "12345"
	layoutID  = "12345"

	partnerLayoutLen    = 5
	nonPartnerLayoutLen = 3
)

var partnerDefaultLayout = map[string]bool{
	"run_ad_1m":          true,
	"edit_stream_info":   true,
	"clip_that":          true,
	"raid_channel":       true,
	"start_squad_stream": true,
}

var nonPartnerDefaultLayout = map[string]bool{
	"edit_stream_info": true,
	"clip_that":        true,
	"raid_channel":     true,
}

func getTestServer(mockdynamo *dynamodb.MockDatabase, mockstats *stats.MockStatSender, mockripley *ripley.MockPayouts) *Server {
	return &Server{
		DynamoDB: mockdynamo,
		Stats:    mockstats,
		Ripley:   mockripley,
	}
}

func TestGetLayoutNoOwnerID(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.GetLayoutReq{
		ChannelID: channelID,
		OwnerID:   "",
	}
	resp, err := server.GetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestGetLayoutNoChannelID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.GetLayoutReq{
		ChannelID: "",
		OwnerID:   ownerID,
	}
	resp, err := server.GetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestGetLayoutDynamoError(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		GetLayout(ctx, gomock.Any(), gomock.Any()).
		Return(dynamomodels.Layout{}, errors.New("error"))

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.GetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
	}
	resp, err := server.GetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestGetLayoutEmptyLayoutNonPartner(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.EXPECT().
		GetLayout(ctx, gomock.Any(), gomock.Any())
	mockDB.
		EXPECT().
		SetLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any())
	mockRipley.
		EXPECT().IsPartner(gomock.Any(), gomock.Any())

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.GetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
	}

	expectedLayout := []*dropship.Item{}
	layout, err := server.GetLayout(ctx, req)
	respLayout := layout.Layout

	for _, respQA := range respLayout {
		if !nonPartnerDefaultLayout[respQA.ID] {
			t.Errorf("%s in response not in non partner default layout", respQA.ID)
		}
	}

	if len(respLayout) != nonPartnerLayoutLen {
		t.Errorf("Expected layout to be len %d got len = %d", nonPartnerLayoutLen, len(respLayout))
	}

	if err != nil {
		t.Errorf("Expected layout to be %s and resp to be nil but got err = %s, resp == %s", expectedLayout, err, layout)
	}
}

func TestGetLayoutEmptyLayoutPartner(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.EXPECT().
		GetLayout(ctx, gomock.Any(), gomock.Any())
	mockDB.
		EXPECT().
		SetLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any())
	mockRipley.
		EXPECT().IsPartner(gomock.Any(), gomock.Any()).
		Return(true, nil)

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.GetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
	}

	layout, err := server.GetLayout(ctx, req)
	respLayout := layout.Layout

	if len(respLayout) != partnerLayoutLen {
		t.Errorf("Expected layout to be len %d got len = %d", partnerLayoutLen, len(respLayout))
	}

	for _, respQA := range respLayout {
		if !partnerDefaultLayout[respQA.ID] {
			t.Errorf("%s in response not in partner default layout", respQA.ID)
		}
	}

	if err != nil {
		t.Errorf("Expected error to be nil but err = %s,=", err)
	}
}

func TestGetLayoutSuccess(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		GetLayout(ctx, gomock.Any(), gomock.Any()).
		Return(dynamomodels.Layout{}, nil)
	mockDB.
		EXPECT().
		SetLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any())
	mockRipley.
		EXPECT().
		IsPartner(gomock.Any(), gomock.Any())

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.GetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
	}
	layout, err := server.GetLayout(ctx, req)
	respLayout := layout.Layout

	for _, respQA := range respLayout {
		if !nonPartnerDefaultLayout[respQA.ID] {
			t.Errorf("%s in response not in layout", respQA.ID)
		}
	}
	if err != nil {
		t.Errorf("Expected err to be nil and resp to be defined but got err = %s, resp == %s", err, layout)
	}
}

func TestSetLayoutInvalidOwnerID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.SetLayoutReq{
		ChannelID: channelID,
		OwnerID:   "",
	}
	resp, err := server.SetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestSetLayoutInvalidChannelID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.SetLayoutReq{
		ChannelID: "",
		OwnerID:   ownerID,
		Layout:    []*dropship.Item{},
	}
	resp, err := server.SetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestSetLayoutInvalidLayout(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.SetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		Layout:    []*dropship.Item{},
	}
	resp, err := server.SetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestSetLayoutDynamoError(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		SetLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any()).
		Return("", errors.New("error"))

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.SetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		Layout:    []*dropship.Item{{Type: "quick_action", ID: "clip_that"}},
	}
	resp, err := server.SetLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestSetLayoutSuccess(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		SetLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any())

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.SetLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		Layout:    []*dropship.Item{{Type: "quick_action", ID: "clip_that"}},
	}
	resp, err := server.SetLayout(ctx, req)
	if err != nil || resp.Layout[0].ID != "clip_that" {
		t.Errorf("Expected err to be nil and resp to be defined but got err = %s, resp == %s", err, resp)
	}
}

func TestUpdateLayoutNoOwnerID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.UpdateLayoutReq{
		ChannelID: channelID,
		OwnerID:   "",
		LayoutID:  layoutID,
		Item:      &dropship.Item{},
	}
	resp, err := server.UpdateLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestUpdateLayoutNoChannelID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.UpdateLayoutReq{
		ChannelID: "",
		OwnerID:   ownerID,
		LayoutID:  layoutID,
		Item:      &dropship.Item{},
	}
	resp, err := server.UpdateLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err = %s, resp == %s", err, resp)
	}
}

func TestUpdateLayoutDynamoError(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		AppendToLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
		Return(errors.New("error"))

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.UpdateLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		LayoutID:  layoutID,
		Item:      &dropship.Item{},
	}
	resp, err := server.UpdateLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err == %s, resp == %s", err, resp)
	}
}

func TestUpdateNoLayoutID(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)

	server := getTestServer(mockDB, mockStats, mockRipley)
	ctx := context.Background()
	req := &pb.UpdateLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		LayoutID:  "",
		Item:      &dropship.Item{},
	}
	resp, err := server.UpdateLayout(ctx, req)
	if err == nil || resp != nil {
		t.Errorf("Expected err to be defined and resp to be nil but got err == %s, resp == %s", err, resp)
	}
}

func TestUpdateLayoutSuccess(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	mockDB := dynamodb.NewMockDatabase(ctrl)
	mockStats := stats.NewMockStatSender(ctrl)
	mockRipley := ripley.NewMockPayouts(ctrl)
	ctx := context.Background()

	mockDB.
		EXPECT().
		AppendToLayout(ctx, gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).
		Return(nil)

	server := getTestServer(mockDB, mockStats, mockRipley)
	req := &pb.UpdateLayoutReq{
		ChannelID: channelID,
		OwnerID:   ownerID,
		LayoutID:  layoutID,
		Item:      &dropship.Item{},
	}

	emptyUpdateResponse := &pb.UpdateLayoutResp{}
	resp, err := server.UpdateLayout(ctx, req)
	if err != nil || resp != emptyUpdateResponse {
		t.Errorf("Expected err to be nil and an resp to be %s but err was %s and resp was %s", emptyUpdateResponse, err, resp)
	}
}
