package quickactions

import (
	"fmt"
	"strings"

	pb "code.justin.tv/cb/dropship/rpc/dropship"
)

const (
	quickActionIDEditStreamInfo   = "edit_stream_info"
	quickActionIDClipThat         = "clip_that"
	quickActionIDRaidChannel      = "raid_channel"
	quickActionIDStartSquadStream = "start_squad_stream"
	quickActionIDRunAd30s         = "run_ad_30s"
	quickActionIDRunAd1m          = "run_ad_1m"
	quickActionIDRunAd2m          = "run_ad_2m"
	quickActionIDStartBounty      = "start_bounty"
	quickActionIDHostChannel      = "host_channel"
	quickActionIDGiftSubs         = "gift_subs"
	quickActionIDGift10Subs       = "gift_10_subs"
	quickActionIDGift50Subs       = "gift_50_subs"
	quickActionIDAddStreamDelay   = "add_stream_delay"
	quickActionIDAddStreamMarker  = "add_stream_marker"
	quickActionIDSchedulePremiere = "schedule_premiere"
	quickActionIDStartRerun       = "start_rerun"
	quickActionIDClearChat        = "clear_chat"
	quickActionIDEmoteOnlyChat    = "emote_only_chat"
	quickActionIDSubOnlyChat      = "sub_only_chat"
	quickActionIDFollowOnlyChat   = "follow_only_chat"
	quickActionIDStartWatchParty  = "start_watch_party"

	typeQuickAction = "quick_action"
	typeFolder      = "folder"

	extensionPrefix = "extensions:"

	maxNestedFoldersAllowed = 8
)

var types = map[string]struct{}{
	typeQuickAction: {},
	typeFolder:      {},
}

var names = map[string]struct{}{
	quickActionIDEditStreamInfo:   {},
	quickActionIDClipThat:         {},
	quickActionIDRaidChannel:      {},
	quickActionIDStartSquadStream: {},
	quickActionIDRunAd30s:         {},
	quickActionIDRunAd1m:          {},
	quickActionIDRunAd2m:          {},
	quickActionIDStartBounty:      {},
	quickActionIDHostChannel:      {},
	quickActionIDGiftSubs:         {},
	quickActionIDGift10Subs:       {},
	quickActionIDGift50Subs:       {},
	quickActionIDAddStreamDelay:   {},
	quickActionIDAddStreamMarker:  {},
	quickActionIDSchedulePremiere: {},
	quickActionIDStartRerun:       {},
	quickActionIDClearChat:        {},
	quickActionIDEmoteOnlyChat:    {},
	quickActionIDSubOnlyChat:      {},
	quickActionIDFollowOnlyChat:   {},
	quickActionIDStartWatchParty:  {},
}

var defaultNamesPartner = map[string]struct{}{
	quickActionIDEditStreamInfo:   {},
	quickActionIDClipThat:         {},
	quickActionIDRaidChannel:      {},
	quickActionIDStartSquadStream: {},
	quickActionIDRunAd1m:          {},
}

var defaultNamesNonPartner = map[string]struct{}{
	quickActionIDEditStreamInfo: {},
	quickActionIDClipThat:       {},
	quickActionIDRaidChannel:    {},
}

type ValidateResult struct {
	InvalidTypes []string
	InvalidNames []string
}

// ValidateLayout reads an entire layout and checks for any names or types that do not exist. It goes into
// each folder provided and checks the IDs in the folder as well. It returns a ValidateResult containing
// lists of invalid names and types. These lists should be checked for invalid strings when evaluating the result.
func ValidateLayout(items []*pb.Item, nestingLevel int) (ValidateResult, error) {
	if nestingLevel > maxNestedFoldersAllowed {
		return ValidateResult{}, fmt.Errorf("validate: found folder nesting greater than the max allowable amount (%d)", maxNestedFoldersAllowed)
	}

	var invalidTypes []string
	var invalidNames []string

	for _, item := range items {

		if strings.HasPrefix(strings.ToLower(item.ID), extensionPrefix) {
			continue
		}

		if len(item.ID) != 0 {
			if _, ok := names[strings.ToLower(item.ID)]; !ok {
				invalidNames = append(invalidNames, item.ID)
			}
		}

		if _, ok := types[strings.ToLower(item.Type)]; !ok {
			invalidTypes = append(invalidTypes, item.Type)
		}

		if item.Type == typeFolder {
			result, err := ValidateLayout(item.IDs, nestingLevel+1)
			if err != nil {
				return result, err
			}
			invalidNames = append(invalidNames, result.InvalidNames...)
		}
	}

	return ValidateResult{
		InvalidTypes: invalidTypes,
		InvalidNames: invalidNames,
	}, nil
}

// GetDefaultNamesForPartners returns the default names of quick actions for partnered streamers
func GetDefaultNamesForPartners() []string {
	var names []string
	for key := range defaultNamesPartner {
		names = append(names, key)
	}
	return names
}

// GetDefaultNamesForNonPartners returns the default names of quick actions for non-partnered streamers
func GetDefaultNamesForNonPartners() []string {
	var names []string
	for key := range defaultNamesNonPartner {
		names = append(names, key)
	}
	return names
}
