package userdestroy

import (
	"context"
	"time"

	"code.justin.tv/cb/dropship/internal/clients/dynamodb"
	"code.justin.tv/cb/dropship/internal/clients/stats"
	"code.justin.tv/eventbus/schema/pkg/user"

	eventbus "code.justin.tv/eventbus/client"
	log "github.com/sirupsen/logrus"
)

// Handler implements github.com/aws/aws-lambda-go/lambda.Handler.
type Handler struct {
	DynamoDB dynamodb.Database
	Statsd   stats.StatSender
}

func (h Handler) Handle(ctx context.Context, header *eventbus.Header, event *user.Destroy) error {
	start := time.Now()
	defer func() {
		since := time.Since(start)
		go h.Statsd.ExecutionTime("userdestroy.duration", since)
	}()

	logger := log.WithField("event", event).WithField("header", header)

	if event == nil {
		logger.Warn("userdestroy: nil eventbus event")
		return nil
	}

	userID := event.GetUserId()
	if userID == "" {
		logger.Warn("userdestroy: userID cannot be empty")
		return nil
	}

	err := h.DynamoDB.DeleteLayout(ctx, userID, userID)
	if err != nil {
		logger.WithError(err).Error("userdestroy: failed to delete layout from dynamodb")
		go h.Statsd.Increment("userdestroy.error", 1)
		return err
	}

	logger.Info("userdestroy: layout destroyed for compliance")
	go h.Statsd.Increment("userdestroy.success", 1)
	return nil
}
