package userdestroy_test

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/cb/dropship/internal/userdestroy"
	"code.justin.tv/eventbus/schema/pkg/user"
	"github.com/golang/mock/gomock"

	dynamodb "code.justin.tv/cb/dropship/internal/mocks/dynamodb"
	stats "code.justin.tv/cb/dropship/internal/mocks/stats"
	log "github.com/sirupsen/logrus"
)

const channelID = "111111"

type mockHandlerParams struct {
	DynamoDB *dynamodb.MockDatabase
	Statsd   *stats.MockStatSender
}

func handlerMocks(ctrl *gomock.Controller) mockHandlerParams {
	return mockHandlerParams{
		DynamoDB: dynamodb.NewMockDatabase(ctrl),
		Statsd:   stats.NewMockStatSender(ctrl),
	}
}

func mockHandler(params mockHandlerParams) *userdestroy.Handler {
	return &userdestroy.Handler{
		DynamoDB: params.DynamoDB,
		Statsd:   params.Statsd,
	}
}

func getInvokeInput(userID string) *user.Destroy {
	return &user.Destroy{
		UserId: userID,
	}
}

func TestInvoke(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()

	mocks := handlerMocks(ctrl)
	handler := mockHandler(mocks)
	ctx := context.Background()

	mocks.Statsd.EXPECT().Increment(gomock.Any(), gomock.Any()).AnyTimes()
	mocks.Statsd.EXPECT().ExecutionTime(gomock.Any(), gomock.Any()).AnyTimes()

	// Test for nil event
	nilEventErr := handler.Handle(ctx, nil, nil)
	if nilEventErr != nil {
		t.Errorf("Invoke: expected nil error, got %s", nilEventErr.Error())
	}

	// Test for nil user ID from event
	event := getInvokeInput("")
	nilUserIDErr := handler.Handle(ctx, nil, event)
	if nilUserIDErr != nil {
		t.Errorf("Invoke: expected nil error, got %s", nilUserIDErr.Error())
	}

	// Test for dynamodb errors
	mocks.DynamoDB.EXPECT().
		DeleteLayout(gomock.Eq(ctx), gomock.Eq(channelID), gomock.Eq(channelID)).
		Return(errors.New("dynamodb error"))

	event = getInvokeInput(channelID)
	ddbErr := handler.Handle(ctx, nil, event)
	if ddbErr == nil {
		t.Error("Invoke: expected dynamodb error, got nil")
	}

	// Test for successful deletion
	mocks.DynamoDB.EXPECT().
		DeleteLayout(gomock.Eq(ctx), gomock.Eq(channelID), gomock.Eq(channelID)).
		Return(nil)

	ddbErr = handler.Handle(ctx, nil, event)
	if ddbErr != nil {
		t.Errorf("Invoke: expected nil error, got %s", ddbErr.Error())
	}
}
