resource "aws_elastic_beanstalk_application" "app" {
  name        = "cb-${var.name}-${var.environment}"
  description = "Go service for getting the layout and positioning of a user's quick actions"
}

resource "aws_elastic_beanstalk_environment" "api" {
  name        = "${aws_elastic_beanstalk_application.app.name}-api"
  application = aws_elastic_beanstalk_application.app.name

  solution_stack_name = var.solution_stack_name
  cname_prefix        = "${aws_elastic_beanstalk_application.app.name}-api"

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team
  }

  lifecycle {
    ignore_changes = [solution_stack_name]
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "DeploymentPolicy"
    value     = "Rolling"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSizeType"
    value     = "Percentage"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSize"
    value     = "50"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name      = "Application Healthcheck URL"
    value     = "/health"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "SystemType"
    value     = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "ConfigDocument"

    value = <<JSON
{
  "Rules": {
    "Environment": {
      "Application": {
        "ApplicationRequests4xx": {
          "Enabled": false
        }
      }
    }
  },
  "CloudWatchMetrics": {
    "Environment": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "InstancesUnknown": 60,
      "InstancesNoData": 60,
      "InstancesSevere": 60,
      "InstancesWarning": 60,
      "InstancesDegraded": 60,
      "InstancesPending": 60,
      "InstancesInfo": 60,
      "InstancesOk": 60
    },
    "Instance": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "CPUIdle": 60,
      "CPUIowait": 60,
      "CPUIrq": 60,
      "CPUNice": 60,
      "CPUSoftirq": 60,
      "CPUSystem": 60,
      "CPUUser": 60,
      "InstanceHealth": 60,
      "LoadAverage1min": 60,
      "LoadAverage5min": 60,
      "RootFilesystemUtil": 60
    }
  },
  "Version": 1
}
JSON

  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "aws-elasticbeanstalk-service-role"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "ENVIRONMENT"
    value     = var.environment
  }

  # Load balancing settings
  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "CrossZone"
    value     = "true"
  }

  setting {
    namespace = "aws:elb:healthcheck"
    name      = "Interval"
    value     = "30"
  }

  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionDrainingEnabled"
    value     = "true"
  }

  # Autoscaling settings
  setting {
    namespace = "aws:autoscaling:asg"
    name      = "Availability Zones"
    value     = "Any 2"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MinSize"
    value     = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MaxSize"
    value     = var.asg_max_size
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "MeasureName"
    value     = "CPUUtilization"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "LowerThreshold"
    value     = "40"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "UpperThreshold"
    value     = "80"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "Unit"
    value     = "Percent"
  }

  # Launch configuration
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = aws_iam_instance_profile.app.id
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = var.security_group
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "EC2KeyName"
    value     = var.aws_key_name
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = "c4.xlarge"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateType"
    value     = "Health"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateEnabled"
    value     = "true"
  }

  # VPC Settings
  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = var.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = var.private_subnets
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "AssociatePublicIpAddress"
    value     = "false"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBScheme"
    value     = "internal"
  }

  setting {
    namespace = "aws:elasticbeanstalk:cloudwatch:logs"
    name      = "StreamLogs"
    value     = "true"
  }

  // Managed updates settings
  setting {
    namespace = "aws:elasticbeanstalk:managedactions"
    name      = "ManagedActionsEnabled"
    value     = "true"
  }

  setting {
    namespace = "aws:elasticbeanstalk:managedactions"
    name      = "PreferredStartTime"
    value     = "Tue:09:00"
  }

  setting {
    namespace = "aws:elasticbeanstalk:managedactions:platformupdate"
    name      = "UpdateLevel"
    value     = "minor"
  }
}
