package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/view"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type createV1EditorsSuite struct {
	suite.Suite
}

func TestCreateV1EditorSuite(t *testing.T) {
	suite.Run(t, &createV1EditorsSuite{})
}

func (s *createV1EditorsSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	params := &view.CreateEditorRequest{
		GrantedBy: "111111",
		GrantedTo: "222222",
	}

	resp, err := client.CreateV1Editor(context.Background(), "999999", params, nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *createV1EditorsSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	params := &view.CreateEditorRequest{
		GrantedBy: "111111",
		GrantedTo: "222222",
	}

	resp, err := client.CreateV1Editor(context.Background(), "999999", params, nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *createV1EditorsSuite) TestSuccess() {
	channelID := "999999"

	payload := `
        {
            "granted_to": "222222"
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/permissions/channels/%s/editors", channelID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	params := &view.CreateEditorRequest{
		GrantedBy: "111111",
		GrantedTo: "222222",
	}

	resp, err := client.CreateV1Editor(context.Background(), channelID, params, nil)
	s.NoError(err)

	if s.NotNil(resp) {
		s.Equal(resp.GrantedTo, "222222")
	}
}
