package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type getV1EditableChannelsSuite struct {
	suite.Suite
}

func TestGetV1EditableChannelsSuite(t *testing.T) {
	suite.Run(t, &getV1EditableChannelsSuite{})
}

func (s *getV1EditableChannelsSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1EditableChannels(context.Background(), "999999", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *getV1EditableChannelsSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1EditableChannels(context.Background(), "999999", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *getV1EditableChannelsSuite) TestSuccess() {
	channelID := "999999"

	payload := `
        {
            "editable": [
                "000000",
                "111111",
                "222222"
            ]
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/permissions/channels/%s/editable_channels", channelID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1EditableChannels(context.Background(), channelID, nil)
	s.NoError(err)

	if s.NotNil(resp) {
		s.Equal(resp.Editable[0], "000000")
		s.Equal(resp.Editable[1], "111111")
		s.Equal(resp.Editable[2], "222222")

	}
}
