package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type getV2EditorsSuite struct {
	suite.Suite
}

func TestGetV2EditorsSuite(t *testing.T) {
	suite.Run(t, &getV2EditorsSuite{})
}

func (s *getV2EditorsSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV2Editors(context.Background(), "999999", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *getV2EditorsSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV2Editors(context.Background(), "999999", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *getV2EditorsSuite) TestSuccess() {
	channelID := "999999"

	payload := `
        {
            "editors": [
                {
                    "editor_user_id": "111111",
                    "created_on": "2018-01-01T10:27:35Z"
                },
                {
                    "editor_user_id": "222222",
                    "created_on": "2018-01-01T10:27:35Z"
                }
            ]
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v2/channels/%s/editors", channelID), r.URL.EscapedPath())
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)
	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV2Editors(context.Background(), channelID, nil)
	s.NoError(err)
	fmt.Println(resp)

	if s.NotNil(resp) {
		s.Equal("111111", resp.Editors[0].EditorUserID)
		s.Equal("222222", resp.Editors[1].EditorUserID)
	}
}
