package main

import (
	"context"
	"net/http"
	"os"
	"strconv"
	"time"

	"code.justin.tv/cb/hallpass/internal/api"
	"code.justin.tv/cb/hallpass/internal/cache"
	"code.justin.tv/cb/hallpass/internal/db"
	"code.justin.tv/cb/hallpass/internal/httputil"
	"code.justin.tv/cb/hallpass/internal/logging"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

const (
	readTimeout  = 5 * time.Second
	writeTimeout = 5 * time.Second
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	logging.SetupRollbar(os.Getenv("ROLLBAR_TOKEN"), os.Getenv("ENVIRONMENT"))
}

func main() {
	maxConns, err := strconv.Atoi(os.Getenv("DB_MAX_CONNS"))
	if err != nil {
		log.WithError(err).Warn("invalid DB_MAX_CONNS")
	}

	dbWriter, err := db.NewClient(db.ClientConfig{
		Host:     os.Getenv("DB_MASTER_HOST"),
		Port:     os.Getenv("DB_MASTER_PORT"),
		Name:     os.Getenv("DB_NAME"),
		User:     os.Getenv("DB_USER"),
		Password: os.Getenv("DB_PASSWORD"),
		SSLMode:  os.Getenv("DB_SSLMODE"),
		MaxConns: maxConns,
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new db client for master connection")
		return
	}

	dbReader, err := db.NewClient(db.ClientConfig{
		Host:     os.Getenv("DB_REPLICA_HOST"),
		Port:     os.Getenv("DB_REPLICA_PORT"),
		Name:     os.Getenv("DB_NAME"),
		User:     os.Getenv("DB_USER"),
		Password: os.Getenv("DB_PASSWORD"),
		SSLMode:  os.Getenv("DB_SSLMODE"),
		MaxConns: maxConns,
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new db client for replica connection")
		return
	}

	redisClient := cache.NewRedisClient(cache.RedisClientConfig{
		Addr:     os.Getenv("REDIS_HOST"),
		Password: "",
		DB:       0,
	})

	usersServiceClient, err := usersclient_internal.NewClient(twitchclient.ClientConf{
		Host: os.Getenv("USERS_SERVICE_HOST"),
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new users service client")
		return
	}

	statsdClient, err := statsd.NewClient(os.Getenv("STATSD_HOST"), os.Getenv("ENVIRONMENT"))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate statsd client")
		return
	}

	server := &http.Server{
		Addr: ":8000",
		Handler: api.NewServer(&api.ServerParams{
			DBReader:     dbReader,
			DBWriter:     dbWriter,
			RedisClient:  redisClient,
			UsersService: usersServiceClient,
			Statsd:       statsdClient,
		}),
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
	}

	go func() {
		log.Info("server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("server failed fatally while listening")
		}
	}()

	httputil.Graceful(context.Background(), server)
}
