package main

import (
	"os"

	"github.com/aws/aws-lambda-go/lambda"

	"code.justin.tv/cb/hallpass/internal/cache"
	"code.justin.tv/cb/hallpass/internal/db"
	"code.justin.tv/cb/hallpass/internal/eventbus/chatban"
	pdmshandler "code.justin.tv/cb/hallpass/internal/eventbus/pdms"
	"code.justin.tv/cb/hallpass/internal/pdms"
	"code.justin.tv/cb/hallpass/internal/secrets"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/subscriber/lambdafunc"
	"code.justin.tv/eventbus/schema/pkg/chat_ban"
	"code.justin.tv/eventbus/schema/pkg/user"
	log "github.com/sirupsen/logrus"

	_ "github.com/lib/pq"
)

func main() {
	secretsManager, err := secrets.NewManager()
	if err != nil {
		log.WithError(err).Fatal("could not create secrets manager: ", err)
		return
	}

	dbPassword, err := secretsManager.GetSecret(os.Getenv("DB_PASSWORD_SECRET"), os.Getenv("DB_PASSWORD_KEY"))
	if err != nil {
		log.WithError(err).Fatal("Unable to retrieve db password.")
		return
	}

	dbWriter, err := db.NewClient(db.ClientConfig{
		Host:     os.Getenv("DB_HOST"),
		Port:     os.Getenv("DB_PORT"),
		Name:     os.Getenv("DB_NAME"),
		User:     os.Getenv("DB_USER"),
		Password: dbPassword,
		SSLMode:  os.Getenv("DB_SSLMODE"),
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new db client for master connection")
		return
	}

	redisClient := cache.NewRedisClient(cache.RedisClientConfig{
		Addr:     os.Getenv("REDIS_HOST"),
		Password: "",
		DB:       0,
	})

	pdmsClient := pdms.NewClient(pdms.ClientConfig{
		CallerRoleArn: os.Getenv("PDMS_CALLER_ROLE_ARN"),
		LambdaArn:     os.Getenv("PDMS_LAMBDA_ARN"),
	})

	pdmsHandler := pdmshandler.Handler{
		DBWriter:    dbWriter,
		RedisClient: redisClient,
		PdmsClient:  pdmsClient,
	}

	chatbanHandler := chatban.Handler{
		DBWriter:    dbWriter,
		RedisClient: redisClient,
	}

	mux := eventbus.NewMux()

	user.RegisterDestroyHandler(mux, pdmsHandler.Handle)
	chat_ban.RegisterCreateHandler(mux, chatbanHandler.Handle)

	lambda.Start(lambdafunc.NewSQS(mux.Dispatcher()))
}
