package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type deleteV1EditorSuite struct {
	suite.Suite
	server       *Server
	dbWriter     *mocks.PermissionsDB
	dbReader     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	UserID       string
	EditorID     string
	Path         string
}

func TestDeleteV1EditorSuite(t *testing.T) {
	suite.Run(t, &deleteV1EditorSuite{})
}

func (s *deleteV1EditorSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.server = NewServer(params)

	s.UserID = "123456"
	s.EditorID = "654321"
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", s.UserID, s.EditorID)
}

func (s *deleteV1EditorSuite) TestWithQueryParam_DBFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodDelete, s.Path, nil)
	s.Require().NoError(err)

	s.dbWriter.On("DeleteEditor", mock.Anything, s.UserID, s.EditorID).Return(nil, errors.New("failed to query editors table"))
	s.usersService.On("GetUserByID", mock.Anything, s.UserID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.EditorID, mock.Anything).Return(nil, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveEditor", s.UserID, s.EditorID)
	s.redisClient.AssertNotCalled(s.T(), "RemoveEditable", mock.Anything, mock.Anything)
	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *deleteV1EditorSuite) TestWithQueryParam_Success() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodDelete, s.Path, nil)
	s.Require().NoError(err)

	s.dbWriter.On("DeleteEditor", mock.Anything, s.UserID, s.EditorID).Return(nil)
	s.redisClient.On("RemoveEditor", s.UserID, s.EditorID).Return(nil)
	s.redisClient.On("RemoveEditable", s.EditorID, s.UserID).Return(nil)

	s.usersService.On("GetUserByID", mock.Anything, s.UserID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.EditorID, mock.Anything).Return(nil, nil)

	s.server.ServeHTTP(recorder, req)

	s.redisClient.AssertExpectations(s.T())
	s.dbReader.AssertExpectations(s.T())

	s.Equal(http.StatusOK, recorder.Code)
}
