package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/cb/hallpass/view"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type getV1EditableChannelsSuite struct {
	suite.Suite
	server       *Server
	dbReader     *mocks.PermissionsDB
	dbWriter     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	ChannelID    string
	Path         string
}

func TestGetV1EditableChannelsSuite(t *testing.T) {
	suite.Run(t, &getV1EditableChannelsSuite{})
}

func (s *getV1EditableChannelsSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.server = NewServer(params)

	s.ChannelID = "123456"
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editable_channels", s.ChannelID)
}

func (s *getV1EditableChannelsSuite) TestWithQueryParam_CacheFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditable", s.ChannelID).Return([]string{}, errors.New("failed to query editors cache"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1EditableChannelsSuite) TestWithQueryParam_DBFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditable", s.ChannelID).Return([]string{}, nil)
	s.dbReader.On("GetEditableChannels", mock.Anything, s.ChannelID).Return([]string{}, errors.New("failed to query editors table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1EditableChannelsSuite) TestWithQueryParam_SuccessWithEditableChannels() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditable", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditable", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditableChannels", mock.Anything, s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditableChannelsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.Editable[0], "654321")
	s.Equal(response.Editable[1], "999999")
}

func (s *getV1EditableChannelsSuite) TestWithQueryParam_SuccessWithNoEditableChannels() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditable", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditable", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditableChannels", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditableChannelsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(len(response.Editable), 0)
}

func (s *getV1EditableChannelsSuite) TestWithQueryParam_SuccessWithEditableChannelsFromCache() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditable", s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertNotCalled(s.T(), "GetEditableChannels", mock.Anything, s.ChannelID)
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditableChannelsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.Editable[0], "654321")
	s.Equal(response.Editable[1], "999999")
}
