package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/cb/hallpass/internal/db"
	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/cb/hallpass/view"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type getV2EditorsSuite struct {
	suite.Suite
	server       *Server
	dbReader     *mocks.PermissionsDB
	dbWriter     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	ChannelID    string
	Path         string
}

func TestGetV2EditorsSuite(t *testing.T) {
	suite.Run(t, &getV2EditorsSuite{})
}

func (s *getV2EditorsSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.server = NewServer(params)

	s.ChannelID = "123456"
	s.Path = fmt.Sprintf("/v2/channels/%s/editors", s.ChannelID)
}

func (s *getV2EditorsSuite) TestDBFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.dbReader.On("GetEditorsWithTimestamps", mock.Anything, s.ChannelID).Return([]db.EditorPermission{}, errors.New("failed to query editors table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV2EditorsSuite) TestSuccessWithEditors() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	creationTime := time.Date(2018, time.January, 1, 1, 0, 0, 0, time.UTC)
	dbResponse := []db.EditorPermission{
		{
			ChannelID:    "123456",
			EditorUserID: "111111",
			CreatedOn:    creationTime,
		},
		{
			ChannelID:    "123456",
			EditorUserID: "222222",
			CreatedOn:    creationTime,
		},
	}

	s.dbReader.On("GetEditorsWithTimestamps", mock.Anything, s.ChannelID).Return(dbResponse, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())

	response := view.V2GetEditorsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(len(response.Editors), 2)
	s.Equal(response.Editors[0].EditorUserID, "111111")
	s.Equal(response.Editors[0].CreatedOn, creationTime)

	s.Equal(response.Editors[1].EditorUserID, "222222")
	s.Equal(response.Editors[1].CreatedOn, creationTime)
}

func (s *getV2EditorsSuite) TestSuccessWithNoEditors() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.dbReader.On("GetEditorsWithTimestamps", mock.Anything, s.ChannelID).Return([]db.EditorPermission{}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())

	response := view.V2GetEditorsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(len(response.Editors), 0)
}
