package api

import (
	"bytes"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/internal/db"
	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	usersServiceModels "code.justin.tv/web/users-service/models"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type createV1EditorSuite struct {
	suite.Suite
	server       *Server
	dbWriter     *mocks.PermissionsDB
	dbReader     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	ChannelID    string
	Path         string
}

func TestCreateV1EditorSuite(t *testing.T) {
	suite.Run(t, &createV1EditorSuite{})
}

func (s *createV1EditorSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.ChannelID = "999999"
	s.server = NewServer(params)
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors", s.ChannelID)
}

func (s *createV1EditorSuite) TestMissingRequestBody() {
	recorder := httptest.NewRecorder()

	req, err := http.NewRequest(http.MethodPost, s.Path, nil)
	s.Require().NoError(err)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "GetCachedEditors", mock.Anything)
	s.usersService.AssertNotCalled(s.T(), "GetUserByID", mock.Anything, mock.Anything, mock.Anything)

	s.Equal(http.StatusBadRequest, recorder.Code)
}

func (s *createV1EditorSuite) TestMissingGrantingToParam() {
	reqBody := []byte(`{ "granted_by":"999999" }`)

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "GetCachedEditors", mock.Anything)
	s.usersService.AssertNotCalled(s.T(), "GetUserByID", mock.Anything, mock.Anything, mock.Anything)

	s.Equal(http.StatusBadRequest, recorder.Code)
}

func (s *createV1EditorSuite) TestMissingGrantedByParam() {
	reqBody := []byte(`{ "granted_to":"999999" }`)

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", mock.Anything)
	s.redisClient.AssertNotCalled(s.T(), "GetCachedEditors", mock.Anything)
	s.usersService.AssertNotCalled(s.T(), "GetUserByID", mock.Anything, mock.Anything, mock.Anything)

	s.Equal(http.StatusBadRequest, recorder.Code)
}

func (s *createV1EditorSuite) TestWithRequestBody_DuplicateEditorID() {
	grantedBy := "999999"
	grantedTo := "111111"

	reqBody := []byte(`{ "granted_by":"999999", "granted_to":"111111" }`)
	currEditors := []string{"111111"}

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return(currEditors, nil)

	s.usersService.On("GetUserByID", mock.Anything, grantedBy, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.ChannelID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, grantedTo, mock.Anything).Return(&usersServiceModels.Properties{}, nil)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return(nil, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.usersService.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", s.ChannelID)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", grantedTo)

	s.Equal(http.StatusUnprocessableEntity, recorder.Code)
}

func (s *createV1EditorSuite) TestWithRequestBody_DBNoEditorCreated() {
	grantedBy := "999999"
	grantedTo := "111111"
	reqBody := []byte(`{ "granted_by":"999999", "granted_to":"111111" }`)

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.dbWriter.On("CreateEditor", mock.Anything, s.ChannelID, grantedTo).Return(db.ErrNoEditorInserted)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.usersService.On("GetUserByID", mock.Anything, grantedBy, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.ChannelID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, grantedTo, mock.Anything).Return(nil, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.usersService.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", grantedBy)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", s.ChannelID)

	s.Equal(http.StatusConflict, recorder.Code)
}

func (s *createV1EditorSuite) TestWithRequestBody_DBError() {
	grantedBy := "999999"
	grantedTo := "111111"
	reqBody := []byte(`{ "granted_by":"999999", "granted_to":"111111" }`)

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.dbWriter.On("CreateEditor", mock.Anything, s.ChannelID, grantedTo).Return(errors.New("failed to query permissions table"))
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.usersService.On("GetUserByID", mock.Anything, grantedBy, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.ChannelID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, grantedTo, mock.Anything).Return(nil, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.usersService.AssertExpectations(s.T())
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditors", grantedBy)
	s.redisClient.AssertNotCalled(s.T(), "RemoveAllEditable", s.ChannelID)

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *createV1EditorSuite) TestWithRequestBody_Success() {
	grantedBy := "999999"
	grantedTo := "111111"
	reqBody := []byte(`{ "granted_by":"999999", "granted_to":"111111" }`)

	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodPost, s.Path, bytes.NewBuffer(reqBody))
	s.Require().NoError(err)

	s.dbWriter.On("CreateEditor", mock.Anything, s.ChannelID, grantedTo).Return(nil)
	s.dbWriter.On("CreateEditor", mock.Anything, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.usersService.On("GetUserByID", mock.Anything, grantedBy, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, s.ChannelID, mock.Anything).Return(nil, nil)
	s.usersService.On("GetUserByID", mock.Anything, grantedTo, mock.Anything).Return(&usersServiceModels.Properties{}, nil)

	s.redisClient.On("RemoveAllEditors", s.ChannelID).Return(nil)
	s.redisClient.On("RemoveAllEditable", grantedTo).Return(nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.dbWriter.AssertExpectations(s.T())
	s.usersService.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusOK, recorder.Code)
}
