package api

import (
	"context"
	"net/http"
	"strconv"

	"code.justin.tv/cb/hallpass/internal/httputil"
	"goji.io/pat"

	usersClient "code.justin.tv/web/users-service/client"
	usersServiceModels "code.justin.tv/web/users-service/models"
)

func (s *Server) validateUser(ctx context.Context, jsonWriter *httputil.JSONResponseWriter, userID string) (*usersServiceModels.Properties, error) {
	user, err := s.usersService.GetUserByID(ctx, userID, nil)
	if err != nil {
		if usersClientError, ok := err.(usersClient.Error); ok {
			switch usersClientError.StatusCode() {
			case http.StatusBadRequest:
				jsonWriter.BadRequest(err.Error())
				return nil, err
			case http.StatusNotFound:
				jsonWriter.NotFound(err.Error())
				return nil, err
			}
		}

		jsonWriter.InternalServerError("Failed to look up granted for channel from users service", err)
		return nil, err
	}

	return user, nil
}

func validateNumericChannelID(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		channelID := pat.Param(req, "channel_id")

		if _, err := strconv.ParseInt(channelID, 10, 32); err != nil {
			httputil.NewJSONResponseWriter(w).BadRequest("provided channel id must be numeric")
			return
		}

		inner.ServeHTTP(w, req)
	}

	return http.HandlerFunc(middleware)
}

func validateNumericEditorID(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		editorID := pat.Param(req, "editor_id")

		if _, err := strconv.ParseInt(editorID, 10, 32); err != nil {
			httputil.NewJSONResponseWriter(w).BadRequest("provided editor id must be numeric")
			return
		}

		inner.ServeHTTP(w, req)
	}

	return http.HandlerFunc(middleware)
}
