package db

import (
	"code.justin.tv/cb/hallpass/internal/postgres"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const defaultMaxConns = 40

// Client contains the postgres db connection information.
type Client struct {
	db *postgres.DB
}

// ClientConfig contains the configuration for the postgres client.
type ClientConfig struct {
	Host     string
	Port     string
	Name     string
	User     string
	Password string
	SSLMode  string
	MaxConns int
}

// NewClient opens a connection pool to the PostgreSQL hallpass database,
// specified by the database credentials and address.
func NewClient(config ClientConfig) (*Client, error) {
	db, err := postgres.NewDB(postgres.DBConfig{
		Host:     config.Host,
		Port:     config.Port,
		Name:     config.Name,
		User:     config.User,
		Password: config.Password,
		SSLMode:  config.SSLMode,
	})
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to instantiate client")
	}

	conns := defaultMaxConns
	if config.MaxConns > 0 {
		conns = config.MaxConns
	}

	db.SetMaxOpenConns(conns)
	db.SetMaxIdleConns(conns)

	log.Infof(
		"db: opened connection to postgresql database at %s:%s/%s",
		config.Host,
		config.Port,
		config.Name,
	)

	return &Client{
		db: db,
	}, nil
}
