package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetEditors returns a list of editors for a given channel id.
func (c *Client) GetEditors(ctx context.Context, channelID string) ([]string, error) {
	query := `
        SELECT channel_id,
               editor_user_id
        FROM   editors
        WHERE  channel_id = $1
    `

	rows, err := c.db.QueryContext(ctx, query, channelID)

	switch {
	case err == sql.ErrNoRows:
		return []string{}, nil
	case err != nil:
		return nil, errors.Wrap(err, "db: failed to scan editors row")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":  query,
				"channel_id": channelID,
			}).Error("db: failed to close rows")
		}
	}()

	return convertRowsToEditors(rows)
}

// GetEditorsWithTimestamps returns a list of editor objects for the given channel id.
func (c *Client) GetEditorsWithTimestamps(ctx context.Context, channelID string) ([]EditorPermission, error) {
	query := `
        SELECT channel_id,
               editor_user_id,
               created_on
        FROM   editors
        WHERE  channel_id = $1
    `

	rows, err := c.db.QueryContext(ctx, query, channelID)

	switch {
	case err == sql.ErrNoRows:
		return []EditorPermission{}, nil
	case err != nil:
		return nil, errors.Wrap(err, "db: failed to scan editors row")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":  query,
				"channel_id": channelID,
			}).Error("db: failed to close rows")
		}
	}()

	editors := []EditorPermission{}

	for rows.Next() {
		permission := EditorPermission{}

		scanErr := rows.Scan(&permission.ChannelID, &permission.EditorUserID, &permission.CreatedOn)
		if scanErr != nil {
			return nil, errors.Wrap(scanErr, "db: failed to scan editors rows")
		}

		editors = append(editors, permission)
	}

	err = rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editors rows")
	}

	return editors, nil
}

// GetEditableChannels returns a list of channels that a given channelID can edit
func (c *Client) GetEditableChannels(ctx context.Context, channelID string) ([]string, error) {
	query := `
        SELECT channel_id,
               editor_user_id
        FROM   editors
        WHERE  editor_user_id = $1
    `

	editors := []string{}
	rows, err := c.db.QueryContext(ctx, query, channelID)

	switch {
	case err == sql.ErrNoRows:
		return editors, nil
	case err != nil:
		return nil, errors.Wrap(err, "db: failed to scan editors row")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":  query,
				"channel_id": channelID,
			}).Error("db: failed to close rows")
		}
	}()

	return convertRowsToEditable(rows)
}

// CreateEditor grants a new editor permission for a user
func (c *Client) CreateEditor(ctx context.Context, channelID string, editorUserID string) error {
	query := `
        INSERT INTO editors
                    (channel_id, editor_user_id)
        SELECT $1, $2
        WHERE NOT EXISTS (
        	SELECT *
        	FROM editors
        	WHERE channel_id = $1
        	AND editor_user_id = $2
        )
    `

	result, err := c.db.ExecContext(ctx, query, channelID, editorUserID)
	if err != nil {
		return errors.Wrap(err, "db: failed to insert new permission")
	}

	rowCount, err := result.RowsAffected()
	if err != nil || rowCount == 0 {
		return ErrNoEditorInserted
	}

	return nil
}

// DeleteEditor removes a single editing permission for a user and an existing editor
func (c *Client) DeleteEditor(ctx context.Context, channelID string, editorID string) error {
	query := `
        DELETE FROM editors
        WHERE       channel_id = $1
        AND         editor_user_id = $2
    `

	_, err := c.db.ExecContext(ctx, query, channelID, editorID)

	if err != nil {
		return errors.Wrap(err, "db: failed to delete permission")
	}

	return nil
}

// DeleteAllEditors removes all editing permissions for a user
func (c *Client) DeleteAllEditors(ctx context.Context, channelID string) error {
	query := `
		DELETE FROM editors
		WHERE channel_id = $1
	`

	_, err := c.db.ExecContext(ctx, query, channelID)

	if err != nil {
		return errors.Wrap(err, "db: failed to delete all editor permissions")
	}

	return nil
}

// DeleteAllEditable removes all editing permissions a user has in other channels
func (c *Client) DeleteAllEditable(ctx context.Context, channelID string) ([]string, error) {
	query := `
		DELETE FROM editors
		WHERE       editor_user_id = $1
		RETURNING   channel_id, editor_user_id
	`

	rows, err := c.db.QueryContext(ctx, query, channelID)

	if err != nil {
		return nil, errors.Wrap(err, "db: failed to delete all editable permissions")
	}

	return convertRowsToEditable(rows)
}

func convertRowsToEditors(rows *sql.Rows) ([]string, error) {
	editors := []string{}

	for rows.Next() {
		permission := EditorPermission{}

		scanErr := rows.Scan(&permission.ChannelID, &permission.EditorUserID)
		if scanErr != nil {
			return nil, errors.Wrap(scanErr, "db: failed to scan editors rows")
		}

		editors = append(editors, permission.EditorUserID)
	}

	err := rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editors rows")
	}

	return editors, nil
}

func convertRowsToEditable(rows *sql.Rows) ([]string, error) {
	editable := []string{}

	for rows.Next() {
		permission := EditorPermission{}

		scanErr := rows.Scan(&permission.ChannelID, &permission.EditorUserID)
		if scanErr != nil {
			return nil, errors.Wrap(scanErr, "db: failed to scan editable rows")
		}

		editable = append(editable, permission.ChannelID)
	}

	err := rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editable rows")
	}

	return editable, nil
}
