# Cannot Terraform a Lambda function without either `filename` or `s3_*`,
# so we have to do a one-time building and zipping of the Go executable in order to apply this Terraform provisioning:
resource "null_resource" "dummy_build" {
  provisioner "local-exec" {
    working_dir = "../../"
    command     = "GOOS=linux go build -o .build/${local.sqs_lambda_handler_name} cmd/dummy/main.go"
  }

  # Locally execute `zip` instead of using `data.archive_file`,
  # so that Terraform does not "read (data resources)" from `data.archive_file` every run.
  provisioner "local-exec" {
    working_dir = "../../.build/"
    command     = "zip ${local.sqs_lambda_handler_name}.zip ${local.sqs_lambda_handler_name}"
  }
}

# Setup the queue for EventBus
data "aws_cloudformation_export" "eventbus_sqs_policy_document" {
  name = "eventbus-sqs-policy-document"
}

data "aws_cloudformation_export" "eventbus_kms_master_key_arn" {
  name = "eventbus-kms-master-key-arn"
}

resource "aws_sqs_queue" "sqs_queue" {
    name                              = var.sqs_queue_name
    policy                            = data.aws_cloudformation_export.eventbus_sqs_policy_document.value
    kms_master_key_id                 = data.aws_cloudformation_export.eventbus_kms_master_key_arn.value
    kms_data_key_reuse_period_seconds = 300
}

resource "aws_sqs_queue" "dlq" {
  name                              = "${var.sqs_queue_name}-dlq"
  # Use one day for DLQ for now, so messages don't dwell and build up for too long of a period.
  message_retention_seconds         = 86400
  kms_master_key_id                 = data.aws_cloudformation_export.eventbus_kms_master_key_arn.value
  kms_data_key_reuse_period_seconds = 300
}

# Setup the lambda
resource "aws_lambda_function" "sqs_handler" {
  depends_on = [
    null_resource.dummy_build,
  ]

  # Note that your function-name must match the name of your Lambda handler name. https://docs.aws.amazon.com/lambda/latest/dg/lambda-go-how-to-create-deployment-package.html
  function_name = local.sqs_lambda_handler_name
  handler       = local.sqs_lambda_handler_name

  description                    = "Handles SQS messages from ${aws_sqs_queue.sqs_queue.arn}"
  filename                       = "../../.build/${local.sqs_lambda_handler_name}.zip"
  role                           = aws_iam_role.pdms_lambda.arn
  runtime                        = "go1.x"

  vpc_config {
    subnet_ids         = split(",", data.terraform_remote_state.account.outputs.private_subnets)
    security_group_ids = [data.terraform_remote_state.account.outputs.twitch_subnets_sg]
  }

  environment {
    variables = {
      PDMS_CALLER_ROLE_ARN = var.pdms_caller_role_arn
      PDMS_LAMBDA_ARN      = var.pdms_lambda_arn
      DB_HOST              = var.db_host
      DB_PORT              = var.db_port
      DB_NAME              = var.db_name
      DB_USER              = var.db_user
      DB_PASSWORD_KEY      = var.db_password_key
      DB_PASSWORD_SECRET   = var.db_password_secret
      DB_SSLMODE           = var.db_sslmode
      REDIS_HOST           = var.redis_host
    }
  }

  tags = {
    environment = var.environment
    name        = var.service_name
    owner       = var.owner
    project     = var.project
    team        = var.team
  }

  dead_letter_config {
    target_arn = aws_sqs_queue.dlq.arn
  }
}

resource "aws_lambda_event_source_mapping" "sqs" {
  batch_size       = 1
  event_source_arn = aws_sqs_queue.sqs_queue.arn
  function_name    = aws_lambda_function.sqs_handler.function_name
}

resource "aws_lambda_permission" "sqs" {
  statement_id  = "allow_sqs_to_invoke"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.sqs_handler.function_name
  principal     = "sqs.amazonaws.com"
  source_arn    = aws_sqs_queue.sqs_queue.arn
}

resource "aws_cloudwatch_log_group" "log_group" {
  name              = "/aws/lambda/${aws_lambda_function.sqs_handler.function_name}"
  retention_in_days = 7
}
