data "aws_iam_policy_document" "assume_lambda_role" {
  statement {
    effect  = "Allow"
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["lambda.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "pdms_lambda" {
  name               = "cb-hallpass-${var.environment}-pdms-lambda"
  assume_role_policy = data.aws_iam_policy_document.assume_lambda_role.json
}

# VPC
resource "aws_iam_role_policy_attachment" "lambda_vpc_access_execution_role" {
  role       = aws_iam_role.pdms_lambda.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

# SQS
resource "aws_iam_role_policy_attachment" "lambda_sqs_queue_execution_role" {
  role       = aws_iam_role.pdms_lambda.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaSQSQueueExecutionRole"
}

# Secrets Manager
resource "aws_iam_role_policy_attachment" "lambda_secrets_manager_access" {
  role       = aws_iam_role.pdms_lambda.name
  policy_arn = "arn:aws:iam::aws:policy/SecretsManagerReadWrite"
}

# Event Bus
data "aws_cloudformation_export" "eventbus_access_policy_arn" {
  name = "eventbus-access-policy-arn"
}

resource "aws_iam_role_policy_attachment" "eventbus_access_attach" {
  role       = aws_iam_role.pdms_lambda.name
  policy_arn = data.aws_cloudformation_export.eventbus_access_policy_arn.value
}

# PDMS
resource "aws_iam_role_policy" "pdms" {
  name   = "allow-pdms-invoke"
  role   = aws_iam_role.pdms_lambda.name
  policy = data.aws_iam_policy_document.pdms_role_policy.json
}

data "aws_iam_policy_document" "pdms_role_policy" {
  version = "2012-10-17"
  statement {
    effect    = "Allow"
    actions   = ["sts:AssumeRole"]
    resources = [var.pdms_caller_role_arn]
  }
}

# DLQ
resource "aws_iam_policy" "dlq_policy" {
  name = "${var.sqs_queue_name}-dlq-send"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "sqs:SendMessage"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
  EOF
}

resource "aws_iam_role_policy_attachment" "dlq_send" {
  role       = aws_iam_role.pdms_lambda.id
  policy_arn = aws_iam_policy.dlq_policy.arn
}

# CloudWatch Logging
resource "aws_iam_policy" "lambda_logging" {
  path        = "/"
  description = "IAM policy for logging from a lambda"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": "arn:aws:logs:*:*:*",
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "lambda_logs" {
  role       = aws_iam_role.pdms_lambda.id
  policy_arn = aws_iam_policy.lambda_logging.arn
}