package main

import (
	"fmt"
	"code.justin.tv/cb/kinesis_processor/config"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"sync"
	"time"
	"code.justin.tv/cb/kinesis_processor/clients/dynamo_adapter"
)


func main() {
	conf := config.LoadConfig()
	dynamoDb, err := dynamo_adapter.NewDynamoAdapter(conf)
	if err != nil {
		panic(err)
	}

	workerPool := NewSessionInsterterWorkerPool(dynamoDb)

	//t1 := time.Now()
	//l, err := clients.Dynamo.GetLastMinutesBroadcast(1)
	//fmt.Println("took ", time.Now().Sub(t1).Seconds(), "seconds")
	//if err != nil {
	//	panic(err)
	//}
	//for _, item := range *l {
	//	fmt.Println(item.ChannelID, ":", item.Time, " ", item.Game)
	//}
	//
	//
	//return
	t1 := time.Now()
	counter := 0
	workerPool.Before()
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	var session *models.ChannelSession

	for {
		list,  newExclusiveStartKey, err := dynamoDb.FetchMinuteBrodcast(exclusiveStartKey)
		exclusiveStartKey = newExclusiveStartKey

		if err != nil {
			panic(err)
		}
		if exclusiveStartKey == nil {
			break
		}

		for _, item := range list {

			if session == nil {
				session = &models.ChannelSession{
					ChannelID: item.ChannelID,
					StartTime: item.Time,
					EndTime: item.Time,
					BroadcastIDs: []int64{item.BroadcastID},
				}

			}

			// Check if we can add to the end otherwise create new session
			if session.ChannelID == item.ChannelID && item.Time.Sub(session.EndTime).Minutes() <= utils.SessionGapMinutes {
				session.EndTime = item.Time
				if !findBroadcast(session.BroadcastIDs, item.BroadcastID) {
					session.BroadcastIDs = append(session.BroadcastIDs, item.BroadcastID)
				}
			} else {
				counter++
				if counter%100 == 0 {
					fmt.Println(time.Now() , "] ", counter , " running for ", time.Now().Sub(t1).Seconds(), "s")
					fmt.Println(exclusiveStartKey)
				}
				workerPool.Process(*session)
				//insertSession(clients, *session)
				session = &models.ChannelSession{
					ChannelID: item.ChannelID,
					StartTime: item.Time,
					EndTime: item.Time,
					BroadcastIDs: []int64{item.BroadcastID},
				}
			}
		}

	}// read all loop
	// Insert last session
	if session != nil {
		workerPool.Process(*session)
		//insertSession(clients, *session)
	}


	workerPool.After()
}

//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//
//
//
//
//
// SessionInserterWorkerPool implementation.
//
//
//
//
//
//


const (
	ParallelInserterWorkersCount = 30
	ParallelInserterBufferSize = 5
)

type SessionInserterWorkerPool interface {
	Before() error
	Process(session models.ChannelSession) error
	After() error
}

type sessionInserterWorkerPool struct {
	wg          sync.WaitGroup
	dynamoDB    dynamo_adapter.DynamoAdapter
	records     chan models.ChannelSession
	deadCounter int
}

func NewSessionInsterterWorkerPool(dynamoDB dynamo_adapter.DynamoAdapter) SessionInserterWorkerPool{

	return &sessionInserterWorkerPool{
		dynamoDB: dynamoDB,
	}
}

// Before - start workers
func (p *sessionInserterWorkerPool) Before() error {
	// Setup worker pool
	p.records = make(chan models.ChannelSession, ParallelInserterWorkersCount)

	for w := 0; w < ParallelInserterWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *sessionInserterWorkerPool) startWorker(number int) {
	p.wg.Add(1)


	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				fmt.Println("Failed worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()


		for model := range p.records {
			err := p.dynamoDB.AddChannelSession(model)
			if err != nil {
				fmt.Println(err)
				break
			}
		}
	}(number)
}

// ProcessEvent
func (p *sessionInserterWorkerPool) Process(session models.ChannelSession) error {
	p.records <- session
	return nil
}

// After
func (p *sessionInserterWorkerPool) After() error {
	close(p.records)
	p.wg.Wait()
	return nil
}
