package main

import (
	"fmt"
	"code.justin.tv/cb/kinesis_processor/config"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"sync"
	"time"
	"os"
	"log"
	"bufio"
	"strconv"
	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/app/processors"
	"code.justin.tv/cb/kinesis_processor/clients/dynamo_adapter"
)

var totalSessionsCounter = 0
var totalChannelsCounter = 0

const (
	ParallelWriterWorkersCount = 25
	ParallelWriterBufferSize   = 25
	ParallelBroadcastReaderWorkersCount = 25
)

func main() {
	workerPool := NewBroadcastReaderWorkerPool()

	workerPool.Before()

	file, err := os.Open("./channel_ids")
	if err != nil {
		panic(err)
	}
	defer file.Close()
	//
	//workerPool.Process(149678932)
	//workerPool.After()
	//return

	t1 := time.Now()
	counter := 0

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		s := scanner.Text()
		channelID, err := strconv.ParseInt(s, 10, 64)
		if err != nil {
			fmt.Println("========Error ", s)
			continue
		}

		workerPool.Process(channelID)
		counter++
		if (counter % 1000 == 0) {
			fmt.Println("=====================================================================")
			fmt.Println("=====================================================================")
			fmt.Println("Processed ", counter, " ", time.Now().Sub(t1))
			fmt.Println("Sessions ", totalSessionsCounter*ParallelWriterBufferSize)
			fmt.Println("Channels read", totalChannelsCounter)
			fmt.Println("=====================================================================")
			fmt.Println("=====================================================================")
			time.Sleep(3*time.Second);
		}
	}

	if err := scanner.Err(); err != nil {
		log.Fatal(err)
	}

	// Wait for workerpool to finish
	workerPool.After()
	fmt.Println("Took ", time.Now().Sub(t1))
}


//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//
//
//
//
//
// SessionInserterWorkerPool implementation.
//
//
//
//
//
//



type SessionWriterWorkerPool interface {
	Before() error
	Process(session models.ChannelSession) error
	After() error
	Flush() error
}

type sessionWriterWorkerPool struct {
	buffer      []models.ChannelSession

	wg          sync.WaitGroup
	records     chan []models.ChannelSession
	deadCounter int
}

func NewSessionWriterWorkerPool() SessionWriterWorkerPool {
	return &sessionWriterWorkerPool{}
}

// Before - start workers
func (p *sessionWriterWorkerPool) Before() error {
	p.buffer = []models.ChannelSession{}

	// Setup worker pool
	p.records = make(chan []models.ChannelSession, ParallelWriterWorkersCount)

	for w := 0; w < ParallelWriterWorkersCount; w++ {
		p.startWorker(w)
		fmt.Println("Starting writer worker #",w )
		time.Sleep(1*time.Second)
	}

	return nil
}

func (p *sessionWriterWorkerPool) startWorker(number int) {
	p.wg.Add(1)


	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				fmt.Println("Failed writer worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()


		conf := config.LoadConfig()
		adapter := adapters.NewChannelSessionAdapter(conf.Environment, conf.AWSRegion)

		for model := range p.records {
			err := adapter.BatchSave(model)
			if err != nil {
				if processors.IsDuplicateError(err) {
					continue
				}
				fmt.Println(err)
				panic(err)
			}
			//fmt.Println(model.ChannelID, ":", model.StartTime, "-", model.EndTime, len(model.BroadcastIDs))
		}
	}(number)
}

// ProcessEvent
func (p *sessionWriterWorkerPool) Process(model models.ChannelSession) error {
	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= ParallelWriterBufferSize {
		totalSessionsCounter++
		err := p.Flush()
		if err != nil {
			fmt.Println(err)
			return err
		}
	}

	return nil
}

// After
func (p *sessionWriterWorkerPool) After() error {
	err := p.Flush()

	close(p.records)
	p.wg.Wait()
	return err
}

func (p *sessionWriterWorkerPool) Flush() error {
	temp := make([]models.ChannelSession, ParallelWriterBufferSize)
	copy(temp, p.buffer)

	p.records <- temp

	p.buffer = []models.ChannelSession{}
	return nil
}

//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//
//
//
//
//
// BroadcastFetcherWorkerPool implementation.
//
//
//
//
//
//


type BroadcastReaderWorkerPool interface {
	Before() error
	Process(channelID int64) error
	After() error
}

type broadcastReaderWorkerPool struct {
	writer      SessionWriterWorkerPool
	wg          sync.WaitGroup
	records     chan int64
	deadCounter int
}

func NewBroadcastReaderWorkerPool() BroadcastReaderWorkerPool {
	return &broadcastReaderWorkerPool{
		writer: NewSessionWriterWorkerPool(),
	}
}

// Before - start workers
func (p *broadcastReaderWorkerPool) Before() error {
	// Setup worker pool
	p.records = make(chan int64, ParallelBroadcastReaderWorkersCount)

	for w := 0; w < ParallelBroadcastReaderWorkersCount; w++ {
		p.startWorker(w)
		fmt.Println("Starting reader worker #",w )
		time.Sleep(1*time.Second)
	}

	p.writer.Before()

	return nil
}

func (p *broadcastReaderWorkerPool) startWorker(number int) {
	p.wg.Add(1)


	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				fmt.Println("Failed reader worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		conf := config.LoadConfig()
		dynamo, err := dynamo_adapter.NewDynamoAdapter(conf)
		if err != nil {
			panic(err)
		}


		// Load specific channel and calculate all the sessions
		for channelID := range p.records {
			var exclusiveStartKey map[string]*dynamodb.AttributeValue
			var session *models.ChannelSession
			counter := 0
			//t1 := time.Now()
			for {
				list,  newExclusiveStartKey, err := dynamo.FetchMinuteBrodcastByChannelID(channelID, exclusiveStartKey)
				exclusiveStartKey = newExclusiveStartKey
				if err != nil {
					panic(err)
				}
				//fmt.Println(channelID, "  ", len(*list))
				for _, item := range *list {
					if session == nil {
						session = &models.ChannelSession{
							ChannelID: item.ChannelID,
							StartTime: item.Time,
							EndTime: item.Time,
							BroadcastIDs: []int64{item.BroadcastID},
						}

					}

					// Check if we can add to the end otherwise create new session
					if item.Time.Sub(session.EndTime).Minutes() <= utils.SessionGapMinutes {
						session.EndTime = item.Time
						if !findBroadcast(session.BroadcastIDs, item.BroadcastID) {
							session.BroadcastIDs = append(session.BroadcastIDs, item.BroadcastID)
						}
					} else {
						counter++
						p.writer.Process(*session)
						//insertSession(clients, *session)
						session = &models.ChannelSession{
							ChannelID: item.ChannelID,
							StartTime: item.Time,
							EndTime: item.Time,
							BroadcastIDs: []int64{item.BroadcastID},
						}
					}
				}

				// Break if this is last chunk
				if exclusiveStartKey == nil {
					break
				}
			}// read all loop
			// Insert last session
			if session != nil {
				counter++
				p.writer.Process(*session)
			}

			//fmt.Println(time.Now() , "] ", channelID , " sessions " ,counter , " running for ", time.Now().Sub(t1).Seconds(), "s")

		}
	}(number)
}

// ProcessEvent
func (p *broadcastReaderWorkerPool) Process(channelID int64) error {
	totalChannelsCounter++

	p.records <- channelID

	return nil
}

// After
func (p *broadcastReaderWorkerPool) After() error {
	close(p.records)
	p.wg.Wait()

	p.writer.After();
	return nil
}

func findBroadcast(broadcasts []int64, broadcastID int64) bool {
	for _, id := range broadcasts {
		if broadcastID == id {
			return true
		}
	}
	return false
}
