package main

import (
	"fmt"
	"code.justin.tv/cb/kinesis_processor/config"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"sync"
	"time"
	"code.justin.tv/cb/kinesis_processor/app/processors"
	"code.justin.tv/cb/kinesis_processor/clients/dynamo_adapter"
	"code.justin.tv/cb/kinesis_processor/clients"
)

//var totalSessionsCounter = 0
var totalChannelsCounter = 0

const (
	ParallelBroadcastMinuteReaderWorkersCount = 25
)

func main() {
	conf := config.LoadConfig()
	clients2 := clients.NewAppClients(conf)
	processor, err := processors.NewChannelSession(clients2)
	if err != nil {
		panic(err)
	}

	dynamo, err := dynamo_adapter.NewDynamoAdapter(config.LoadConfig())
	if err != nil {
		panic(err)
	}

	t1 := time.Now()
	counter := 0
	BEFORE_TIME, _  := time.Parse(utils.DbTimeFormat, "2017-08-16 23:43:41")

	channelBasedList := []models.MinuteBroadcast{}
	var currentChannelID int64
	currentChannelID = 0

	processor.Before()
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	for {
		list,  newExclusiveStartKey, err := dynamo.FetchMinuteBrodcast(exclusiveStartKey)
		exclusiveStartKey = newExclusiveStartKey
		if err != nil {
			fmt.Println(err)
			time.Sleep(3*time.Second)
			continue
		}

		for _, item := range list {
			// Ignore every minute_broadcast after BEFORE_TIME
			if BEFORE_TIME.Sub(item.Time) < 0 {
				//continue
			}
			if currentChannelID == 0 {
				currentChannelID = item.ChannelID
				channelBasedList = []models.MinuteBroadcast{}
			}

			if currentChannelID != item.ChannelID {
				fmt.Println("Processed ", len(channelBasedList), "/", counter, " ", currentChannelID," ", time.Now().Sub(t1))

				processor.Process(channelBasedList)
				// reset
				channelBasedList = []models.MinuteBroadcast{}
				currentChannelID = item.ChannelID


			}
			if counter%1000 == 0{
				fmt.Println("===================================================")
				fmt.Println("Processed ", counter, " last key ", exclusiveStartKey)
				fmt.Println("===================================================")
			}
			counter++
			channelBasedList = append(channelBasedList, item)
		}

		// Break if this is last chunk
		if exclusiveStartKey == nil {
			break
		}
	}// read all loop
	fmt.Println("Processed ", counter, " ", currentChannelID," ", time.Now().Sub(t1))
	processor.Process(channelBasedList)

	processor.After()
	fmt.Println("Took ", time.Now().Sub(t1))
}


//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------------------
//
//
//
//
//
// BroadcastMinuteReaderWorkerPool implementation.
//
//
//
//
//
//


type BroadcastMinuteReaderWorkerPool interface {
	Before() error
	Process(channelID int64) error
	After() error
}

type broadcastMinuteReaderWorkerPool struct {
	processor   processors.ChannelSession
	wg          sync.WaitGroup
	records     chan int64
	deadCounter int
}

func NewBroadcastMinuteReaderWorkerPool() BroadcastMinuteReaderWorkerPool {
	clients2 := clients.NewAppClients(config.LoadConfig())
	processor, err := processors.NewChannelSession(clients2)
	if err != nil {
		panic(err)
	}
	return &broadcastMinuteReaderWorkerPool{
		processor: processor,
	}
}

// Before - start workers
func (p *broadcastMinuteReaderWorkerPool) Before() error {
	// Setup worker pool
	p.records = make(chan int64, ParallelBroadcastMinuteReaderWorkersCount)

	for w := 0; w < ParallelBroadcastMinuteReaderWorkersCount; w++ {
		p.startWorker(w)
		fmt.Println("Starting reader worker #",w )
		time.Sleep(1*time.Second)
	}
	p.processor.Before()

	return nil
}

func (p *broadcastMinuteReaderWorkerPool) startWorker(number int) {
	p.wg.Add(1)


	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				fmt.Println("Failed reader worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		dynamo, err := dynamo_adapter.NewDynamoAdapter(config.LoadConfig())
		if err != nil {
			panic(err)
		}
		BEFORE_TIME, _  := time.Parse(utils.DbTimeFormat, "2017-08-16 23:43:41")

		// Load specific channel and calculate all the sessions
		for channelID := range p.records {
			var exclusiveStartKey map[string]*dynamodb.AttributeValue
			counter := 0
			t1 := time.Now()
			for {
				list,  newExclusiveStartKey, err := dynamo.FetchMinuteBrodcastByChannelID(channelID, exclusiveStartKey)
				exclusiveStartKey = newExclusiveStartKey
				if err != nil {
					panic(err)
				}
				//fmt.Println(channelID, "  ", len(*list))
				newList := []models.MinuteBroadcast{}

				for _, item := range list {
					// Ignore every minute_broadcast after BEFORE_TIME
					if BEFORE_TIME.Sub(item.Time) < 0 {
						continue
					}
					newList = append(newList, item)
				}
				p.processor.Process(newList)

				// Break if this is last chunk
				if exclusiveStartKey == nil {
					break
				}
			}// read all loop

			fmt.Println(time.Now() , "] ", channelID , " minutes " ,counter , " running for ", time.Now().Sub(t1))

		}
	}(number)
}

// ProcessEvent
func (p *broadcastMinuteReaderWorkerPool) Process(channelID int64) error {
	totalChannelsCounter++

	p.records <- channelID

	return nil
}

// After
func (p *broadcastMinuteReaderWorkerPool) After() error {
	close(p.records)
	p.wg.Wait()

	p.processor.After()
	return nil
}