package main

import (
	"code.justin.tv/cb/kinesis_processor/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/aws/session"
	"code.justin.tv/cb/kinesis_processor/adapters"
	"fmt"
	"time"
	"sync"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/aws/credentials"
)

func main() {
	BEFORE_TIME, _  := time.Parse(utils.DbTimeFormat, "2017-08-16 23:43:41")


	conf := config.LoadConfig()
	creds := credentials.NewSharedCredentials("", "twitch-web-dev")

	awsConfig := &aws.Config{
		Region:           aws.String(conf.AWSRegion),
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
	}

	ddb := dynamodb.New(session.New(awsConfig))
	pool := NewSessionDeleteWorkerPool2(ddb)
	pool.Before()
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	counter := 0
	processed := 0
	for {

		output, err := ddb.Scan(&dynamodb.ScanInput{

			TableName:                 aws.String(adapters.TableChannelSessions),
			ExclusiveStartKey:         exclusiveStartKey,
			Limit:                     aws.Int64(100),
		})

		if err != nil {
			panic(err)
		}

		for _, value := range output.Items {
			endTime, err := time.Parse(utils.DbTimeFormat, *value["EndTime"].S)
			if err != nil {
				fmt.Println(err)
				continue
			}

			if processed % 1000 == 0 {
				fmt.Println("Deleted ", processed, " / ", counter, " ", time.Now());
			}

			if BEFORE_TIME.Sub(endTime).Seconds() > 0 {
				err = pool.Process(value)
				processed++

			}
			counter++
			if err != nil {
				fmt.Println(value)
				panic(err)
			}
		}

		if (output.LastEvaluatedKey == nil) {
			break
		}
		exclusiveStartKey = output.LastEvaluatedKey
	}
	pool.After()
	fmt.Println("Deleted ", processed, "/", counter)
}



const (
	ParallelSessionDeleteWorkersCount2 = 50
)

type SessionDeleteWorkerPool2 interface {
	Before() error
	Process(value map[string]*dynamodb.AttributeValue) error
	After() error
}

type sessionDeleteWorkerPool2 struct {
	client     *dynamodb.DynamoDB
	wg          sync.WaitGroup
	records     chan map[string]*dynamodb.AttributeValue
	deadCounter int
}

func NewSessionDeleteWorkerPool2(client *dynamodb.DynamoDB) SessionDeleteWorkerPool2{
	return &sessionDeleteWorkerPool2{
		client: client,
	}
}

// Before - start workers
func (p *sessionDeleteWorkerPool2) Before() error {
	// Setup worker pool
	p.records = make(chan map[string]*dynamodb.AttributeValue, ParallelSessionDeleteWorkersCount2)

	for w := 0; w < ParallelSessionDeleteWorkersCount2; w++ {
		p.startWorker(w)
	}


	return nil
}

func (p *sessionDeleteWorkerPool2) startWorker(number int) {
	p.wg.Add(1)


	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				fmt.Println("Failed delete worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		// Load specific channel and calculate all the sessions
		for value := range p.records {
			_, err := p.client.DeleteItem(&dynamodb.DeleteItemInput{
				TableName: aws.String(adapters.TableChannelSessions),
				Key: map[string]*dynamodb.AttributeValue{
					"ChannelID": {
						N: value["ChannelID"].N,
					},
					"StartTime": {
						S: value["StartTime"].S,
					},
				},

			})
			if err != nil{
				panic(err)
			}
		}
	}(number)
}

// ProcessEvent
func (p *sessionDeleteWorkerPool2) Process(value map[string]*dynamodb.AttributeValue) error {
	p.records <- value

	return nil
}

// After
func (p *sessionDeleteWorkerPool2) After() error {
	close(p.records)
	p.wg.Wait()

	return nil
}