package main

import (
	"encoding/json"
	"fmt"
	"compress/flate"
	"bytes"
	"io"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/config"
	"code.justin.tv/cb/kinesis_processor/models"
	"github.com/jmoiron/sqlx"
	"strings"
	"time"
	"bufio"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/aws/session"
	"code.justin.tv/cb/kinesis_processor/utils"
	"math"
	"code.justin.tv/cb/kinesis_processor/adapters"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"strconv"
	"context"
)

// KinesisRecord is the JSON object returned from Kinsumer.Next().
type KinesisRecord struct {
	UUID    string
	Version int
	Data    []byte
}

// KinesisEvent is used to unmarshal spade events from a KinesisRecord.
type KinesisEvent struct {
	Name   string
	Fields map[string]string
}

// ========================================================================================================================
// ========================================================================================================================
// ========================================================================================================================
func main() {
	//mainTestUpdateSessions()
	//mainTestSelectChannelSession()
	//mainTestSession_BatchSave()
	//mainCalculateSessions()
	mainLastSessions()
	//mainDeleteAll()
}
// ========================================================================================================================
// ========================================================================================================================
// ========================================================================================================================


func main1() {
	f := map[string]string{
		"test":"test",
	}

	fmt.Println(f["test"])
	fmt.Println(f["test1"] == "")
}

func main2() {
	fmt.Println(math.Floor(55/5)*5)
}

func mainTime() {
	loc, err := time.LoadLocation("America/Los_Angeles")
	if err != nil { panic(err) }

	t, err := time.ParseInLocation("2006-01-02 15:04:05", "2016-12-07 16:43:16", loc)
	if err != nil { panic(err) }

	fmt.Println(t.UTC().Format("2006-01-02 15:04:05"))
}

func mainD() {
	//s := "{\"UUID\":\"60da8c15-7222-40af-8de2-d39af6b53a65\",\"Version\":1,\"Data\":\"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\",\"CreatedAt\":\"\"}"
	s := `
	{"UUID":"e0b67f52-df4f-402d-8307-d2f3a0d14e85","Version":1,"Data":"AaSQQU7DMBBFr2J5w6aV4mDSKFsqKiRgA4IFQtE0maSWHAc5E6FQ9QycgB2H4DxcgCtghzTphm7Y2fM9/v/9xy2/gQp5wrMNGIM6bZ9zIOQzfqFQ5w1Ptnxta8gzaAhtqsGULZR+w70ZltxlXWWZbtFOw1Tlbi6kkJGQMnBC+et03bGVRSC28vcZr3We/m3h1WFxeNsQUOuC+QDj+Q4bUqZkt+S+rpxCqscKAxHNRTgPFkxEiQwTEfPdbvYPbDTu9wlc0asBAq3Qs+xb7NkXcSTPRHDq5gPBsvMZr1S5ITc8At6bHKCfg9asLtiypS5hl6ZQRhGyB7AF2H2JYxn3Cl/QsvikYV/vH9+fb85sFA8yTOrRtp5+AgAA//8=","CreatedAt":""}
`

	lines := strings.Split(s, "\n")
	for _, line := range lines {
		var kinesisRecord KinesisRecord

		err := json.Unmarshal([]byte(line), &kinesisRecord)
		if err != nil {
			fmt.Println("json Unmarshal error: %v", err)
			continue
		}

		if kinesisRecord.Version != 1 {
			fmt.Println("kinesis event version was not 1, instead: %v", kinesisRecord.Version)
		}

		if kinesisRecord.Data[0] != 1 {
			fmt.Println("data[0] was not 1, instead: %v", kinesisRecord.Data[0])
		}

		deflator := flate.NewReader(bytes.NewBuffer(kinesisRecord.Data[1:]))
		defer func() {
			err = deflator.Close()
			if err != nil {
				fmt.Println("Error closing deflator: %v", err)
			}
		}()

		var decompressed bytes.Buffer
		_, err = io.Copy(&decompressed, deflator)
		if err != nil {
			fmt.Println("Error decompressing: %v", err)
		}

		var kinesisEvents []KinesisEvent
		err = json.Unmarshal(decompressed.Bytes(), &kinesisEvents)
		if err != nil {
			fmt.Println("json Unmarshal error: %v", err)
		}

		for _, event := range kinesisEvents {
			fmt.Println(event)
		}
	}
}

func mainD2() {
	conf := config.LoadConfig()
	credentials := conf.NewAWSCredentials()

	awsConfig := &aws.Config{
		Region:           aws.String(conf.AWSRegion),
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      credentials,
	}

	client := s3.New(session.New(awsConfig))

	prefix := "failed_incomming/2016/12/22/19/cb-analytics-1-2016-12-22-19-05"

	// get first object out there
	listInput := &s3.ListObjectsV2Input{
		Bucket:  aws.String(conf.FirehoseS3Bucket),
		Prefix:  aws.String(prefix),
	}
	resp, err := client.ListObjectsV2(listInput)
	if err != nil {
		panic(err)
	}

	// if no keys returned
	if *resp.KeyCount == 0 {
		fmt.Println("no keys found")
		return
	}

	fmt.Println("Objects found ", len(resp.Contents))

	for _, item := range resp.Contents {
		objectOutput, err := client.GetObject(&s3.GetObjectInput{
			Bucket: aws.String(conf.FirehoseS3Bucket),
			Key:    item.Key,
		})
		if err != nil {
			panic(err)
		}

		scanner := bufio.NewScanner(objectOutput.Body)
		buf := make([]byte, 0, 64 * 1024)
		scanner.Buffer(buf, 1024 * 1024)
		for scanner.Scan() {
			// Read Kinesis Record
			var kinesisRecord KinesisRecord
			err := json.Unmarshal(scanner.Bytes(), &kinesisRecord)
			if err != nil {
				panic(err)
			}

			// Validate version
			if kinesisRecord.Version != 1 {
				panic(err)
			}

			// Validate first byte of data == 1
			if kinesisRecord.Data[0] != 1 {
				panic(err)
			}

			// Read KinesisEvents from KinesisRecord.Data
			deflator := flate.NewReader(bytes.NewBuffer(kinesisRecord.Data[1:]))
			defer func() {
				err = deflator.Close()
				if err != nil {
					panic(err)
				}
			}()

			var decompressed bytes.Buffer
			_, err = io.Copy(&decompressed, deflator)
			if err != nil {
				panic(err)
			}

			var kinesisEvents []KinesisEvent
			err = json.Unmarshal(decompressed.Bytes(), &kinesisEvents)
			if err != nil {
				panic(err)
			}

			for _, event := range kinesisEvents {
				//if event.Name == "minute_broadcast" && event.Fields["channel_id"] == "89799292"{
				//	fmt.Println(event)
				//}
					fmt.Println(event)

			}
		}
	}
}


func mainReadFromS3() {
	conf := config.LoadConfig()
	firehose := kinesis_firehose.NewKinesisFirehose(conf,  conf.FirehoseS3Bucket, conf.FirehoseS3Prefix)

	result, err := firehose.Get()

	if err != nil {
		fmt.Println(err)
		return
	}

	fmt.Println(result.Key)
	for i, o := range(result.Events) {
		fmt.Println(i, o)
	}
}


func mainTestSession_BatchSave() {
	conf := config.LoadConfig()

	adapter, err := adapters.NewChannelSessionAdapter("development", conf.AWSRegion)
	if err != nil {
		panic(err)
	}

	t, err := time.Parse(utils.DbTimeFormat, "2016-12-05 20:49:10")
	if err != nil {
		panic(err)
	}

	t2, err := time.Parse(utils.DbTimeFormat, "2016-12-05 22:49:10")
	if err != nil {
		panic(err)
	}

	model := models.ChannelSession{
		ChannelID: 123,
		StartTime: t,
		EndTime: t2,
		BroadcastIDs: []int64{ 123, 345, 674},
	}

	err = adapter.BatchSave([]models.ChannelSession{model})
	if err != nil {
		panic(err)
	}

}

func mainTestDynamoSave() {
	conf := config.LoadConfig()

	adapter := adapters.NewChannelConcurrentAdapter("development", conf.AWSRegion)

	t, err := time.Parse(utils.DbTimeFormat, "2016-12-05 20:49:10")
	if err != nil {
		panic(err)
	}

	model := models.ChannelConcurrent{
		ChannelID: 123,
		Time: t,
		ContentMode: "live",
		Total: 123,
	}

	err = adapter.Save(context.TODO(), model)
	if err != nil {
		panic(err)
	}
}

func mainTestUpdateSessions() {
	conf := config.LoadConfig()
	db := newDB(conf)

	startTime, err := time.Parse(utils.DbTimeFormat, "2016-12-05 20:49:10")
	if err != nil {
		panic(err)
	}


	endTime, err := time.Parse(utils.DbTimeFormat, "2016-12-05 20:50:10")
	if err != nil {
		panic(err)
	}


	model := models.ChannelSession{
		//ID: 22696786,
		ChannelID: 111997428,
		StartTime: startTime,
		EndTime: endTime,
	}
	fmt.Println(model)

	//result, err := db.NamedExec(
	//	"UPDATE channel_sessions SET end_time = :end_time AND start_time = :start_time AND channel_id = :channel_id WHERE id = :id",
	//	model,
	//)
	result, err := db.Exec("UPDATE channel_sessions SET start_time = ?, end_time = ?, channel_id = ? WHERE id = ?",
		"2016-12-05 20:49:09", "2016-12-05 20:50:10", 111997428, 22696786)

	if err != nil {
		panic(err)
	}

	rows, err := result.RowsAffected()
	if err != nil {
		panic(err)
	}
	fmt.Println(rows)
}


func mainTestConcurrentRange() {
	client := newDynamoDB()

	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String("111997428"),
		},
		":startTime": {
			S: aws.String("2016-12-22 00:00:00"),
		},
		":endTime": {
			S: aws.String("2016-12-23 00:00:00"),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := client.Query(&dynamodb.QueryInput{
		TableName:                 aws.String("CbChannelConcurrents"),
		ScanIndexForward:          aws.Bool(false),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		panic(err)
	}

	fmt.Println(*output.Count)
	fmt.Println(output.Items)
}

func mainDeleteAll() {
	client := newDynamoDB()
	/*
		output, err := client.DeleteItem(&dynamodb.DeleteItemInput{
		TableName:                 aws.String("CbChannelConcurrents"),
		Key: ,
	})

	 */
	var exlusiveStartKey map[string]*dynamodb.AttributeValue

	for {
		output, err := client.Scan(&dynamodb.ScanInput{
			TableName:                 aws.String("CbMinuteBroadcasts"),
			Limit:                     aws.Int64(1000),
			ExclusiveStartKey:         exlusiveStartKey,
		})

		if err != nil {
			panic(err)
		}

		if *output.Count == 0 {
			fmt.Println("output.Count == 0")
			break
		}
		fmt.Println("[",time.Now().Format(utils.DbTimeFormat),"] Deleting ", *output.Count)


		for _, value := range output.Items {
			go func() {
				key := map[string]*dynamodb.AttributeValue{
					"ChannelID": {
						N: value["ChannelID"].N,
					},
					"Time": {
						S: value["Time"].S,
					},
				}

				_, err := client.DeleteItem(&dynamodb.DeleteItemInput{
					TableName:                 aws.String("CbMinuteBroadcasts"),
					Key: key,
				})

				if err != nil {
					fmt.Println(err)
				}
			}()
		}

		time.Sleep(1*time.Second)

		if err != nil {
			panic(err)
		}
		exlusiveStartKey = output.LastEvaluatedKey
	}

	fmt.Println("Done")
}


func mainCalculateSessions() {
	client := newDynamoDB()
	size := 10
	var exlusiveStartKey map[string]*dynamodb.AttributeValue

	sessions := []models.ChannelSession{}
	for {
		keyCondition := aws.String("ChannelID = :channelID")
		conditionAttrValues := map[string]*dynamodb.AttributeValue{
			":channelID": {
				N: aws.String("40017619"),
			},
		}

		output, err := client.Query(&dynamodb.QueryInput{
			TableName:                   aws.String("CbMinuteBroadcasts"),
			ScanIndexForward:            aws.Bool(true),
			Limit:                       aws.Int64(1000),
			KeyConditionExpression:      keyCondition,
			ExpressionAttributeValues:   conditionAttrValues,
			ExclusiveStartKey:           exlusiveStartKey,
		})

		if err != nil {
			panic(err)
		}

		if *output.Count == 0 {
			fmt.Println("output.Count == 0")
			break
		}

		currentSession := models.ChannelSession{ChannelID: 0}
		if len(sessions) > 0 {
			currentSession = sessions[len(sessions) - 1]
		}

		for _, value := range output.Items {
			channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
			if err != nil {
				panic(err)
			}
			broadcastIDStr := value["BroadcastID"]
			if broadcastIDStr == nil {
				broadcastIDStr = value["BroadcasterID"]
			}
			broadcastID, err := strconv.ParseInt(*broadcastIDStr.N, 10, 64)
			if err != nil {
				panic(err)
			}

			timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
			if err != nil {
				panic(err)
			}
			fmt.Println(timetime, broadcastID)

			if currentSession.ChannelID != channelID || timetime.Sub(currentSession.EndTime) >= 10*time.Minute {
				if currentSession.ChannelID != 0 {
					sessions = append(sessions, currentSession)

					if len(sessions) >= size {
						fmt.Println(sessions)
						return
					}
				}

				currentSession = models.ChannelSession{
					ChannelID: channelID,
					StartTime: timetime,
					EndTime: timetime,
				}
				continue
			}

			currentSession.EndTime = timetime
		}
		sessions = append(sessions, currentSession)

		if err != nil {
			panic(err)
		}
		if output.LastEvaluatedKey == nil {
			break
		}
		exlusiveStartKey = output.LastEvaluatedKey
	}
	for _, item := range sessions {
		fmt.Println("ChannelID",item.ChannelID,"StartTime:", item.StartTime,"EndTime:",item.EndTime)
	}

	fmt.Println("DOne")
}

func mainLastSessions() {
	/*
	lirik 23161357
	theokoles 46864514
	toddawesome 58697170
	nightblue3 26946000
	qa user 192608
	srkevo1 30917811
	imls 26513896
	voyboy 14293484
	tashnarr 76962228 12036
	doublelift 40017619
	anniebot 44019612
	imaqtpie 24991333
	24/7 streamerhouse 44741426
	24/7 monsterhouse 27446517

	defsantv 111997428
	notfrank 20624989
	mrchowderclam 	50707402
	2chengz 103838077
	 */
	//
	//adapter := adapters.NewMinuteBroadcastAdapter("development", "us-west-2")
	//sessions, err := adapter.CalculateLastSessions(7883503, 100)
	channelID := int64(116818637)
	adapter, err := adapters.NewChannelSessionAdapter("development", "us-west-2")
	adapter2 := adapters.NewMinuteBroadcastAdapter("development", "us-west-2")

	sessions, err := adapter.GetLastList(context.TODO(), channelID, 20)
	if err != nil {
		panic(err)
	}

	sessions2, err := adapter2.CalculateLastSessions(context.TODO(), channelID, 20)
	if err != nil {
		panic(err)
	}

	for _, item := range sessions {
		fmt.Println("===ChannelID", item.ChannelID, "StartTime:", item.StartTime, "EndTime:", item.EndTime, "BroadcastIDS", item.BroadcastIDs)
	}
	fmt.Println("========================================================================================")
	for _, item := range sessions2 {
		fmt.Println("===ChannelID", item.ChannelID, "StartTime:", item.StartTime, "EndTime:", item.EndTime, "BroadcastIDS", item.BroadcastIDs)
	}

}

//
//
//
//
//
//
// HELPER FUNCTIONS
///
//
//
//
//
//

func newDB(conf *config.Config) *sqlx.DB {
	dbDataSource := conf.DBDataSource()

	db, err := sqlx.Open("mysql", dbDataSource)
	if err != nil {
		panic(err)
	}
	return db
}

func newDynamoDB() *dynamodb.DynamoDB{
	creds := helper.NewCredentials("development", "us-west-2")
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String("us-west-2"),
	}

	return dynamodb.New(session.New(awsConfig))
}

