package adapters

import (
	"context"
	"math"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableChannelConcurrents = "CbChannelConcurrents"
)

//
// ChannelConcurrentAdapter processor.
//
type ChannelConcurrentAdapter interface {
	Save(ctx context.Context, models models.ChannelConcurrent) error
	// BatchSave - saves model into DynamoDatabase under specific key
	// defined by channel_id and time defined in model.
	BatchSave(ctx context.Context, models []models.ChannelConcurrent) error
	// GetAllByTime  return ChannelConcurrents given
	// channelID and startTime and endTime.
	// Returns []models.ChannelConcurrent if found and error
	// if something went wrong.
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ChannelConcurrent, error)
}

type channelConcurrentAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewChannelConcurrentAdapter create new processor.
func NewChannelConcurrentAdapter(env string, region string) ChannelConcurrentAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &channelConcurrentAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// Save - saves model as JSON into S3 under specific key
// defined by channel_id and time defined in model.
func (c *channelConcurrentAdapter) Save(ctx context.Context, model models.ChannelConcurrent) error {

	minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	_, err := c.client.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"ChannelID":   &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":        &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
			"Total":       &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.Total, 10))},
			"ContentMode": &dynamodb.AttributeValue{S: aws.String(model.ContentMode)},
		},
		TableName: aws.String(TableChannelConcurrents),
	})

	return err
}

// BatchSave - saves model into DynamoDatabase under specific key
// defined by channel_id and time defined in model.
func (c *channelConcurrentAdapter) BatchSave(ctx context.Context, models []models.ChannelConcurrent) error {
	if len(models) == 0 {
		return nil
	}

	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			TableChannelConcurrents: make([]*dynamodb.WriteRequest, len(models)),
		},
		ReturnConsumedCapacity: aws.String(dynamodb.ReturnConsumedCapacityTotal),
	}

	for i, model := range models {
		minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
		flatTime := time.Date(model.Time.Year(), model.Time.Month(),
			model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

		//model.Time
		item := map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
			"Total":     &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.Total, 10))},
		}

		if model.ContentMode != "" {
			item["ContentMode"] = &dynamodb.AttributeValue{S: aws.String(model.ContentMode)}
		}

		input.RequestItems[TableChannelConcurrents][i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: item,
			},
		}
	}

	_, err := c.client.BatchWriteItemWithContext(ctx, input)

	return err
}

// GetAllByTime  return ChannelConcurrents given
// channelID and startTime and endTime.
// Returns []models.ChannelConcurrent if found and error
// if something went wrong.
func (c *channelConcurrentAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ChannelConcurrent, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableChannelConcurrents),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	result := make([]models.ChannelConcurrent, *output.Count)
	for ind, value := range output.Items {
		timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
		if err != nil {
			return nil, err
		}

		total, err := strconv.ParseInt(*value["Total"].N, 10, 64)
		if err != nil {
			return nil, err
		}

		mode := ""
		if value["ContentMode"] != nil {
			mode = *value["ContentMode"].S
		}

		result[ind] = models.ChannelConcurrent{
			ChannelID:   channelID,
			Time:        timetime,
			Total:       total,
			ContentMode: mode,
		}
	}

	return result, nil
}
