package adapters

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableChannelUpdates = "CbChannelUpdates"
)

//
// ChannelUpdateAdapter processor.
//
type ChannelUpdateAdapter interface {
	// BatchSave - saves model into DynamoDatabase under specific key
	// defined by channel_id and time defined in model.
	BatchSave(ctx context.Context, models []models.ChannelUpdate) error

	// GetAllByTime return ChannelUpdates given
	// channelID and startTime and endTime.
	// Returns []models.ChannelUpdate if found and error
	// if something went wrong.
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ChannelUpdate, error)

	// GetFirstBefore return last ChannelUpdates before given time.
	// Returns models.ChannelUpdate if found and error
	// if something went wrong.
	GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.ChannelUpdate, error)
}

type channelUpdateAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewChannelUpdateAdapter create new processor.
func NewChannelUpdateAdapter(env string, region string) ChannelUpdateAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &channelUpdateAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// BatchSave - saves model into DynamoDatabase under specific key
// defined by channel_id and time.
func (c *channelUpdateAdapter) BatchSave(ctx context.Context, models []models.ChannelUpdate) error {
	if len(models) == 0 {
		return nil
	}

	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			TableChannelUpdates: make([]*dynamodb.WriteRequest, len(models)),
		},
		ReturnConsumedCapacity: aws.String(dynamodb.ReturnConsumedCapacityTotal),
	}

	for i, model := range models {
		flatTime := time.Date(model.Time.Year(), model.Time.Month(),
			model.Time.Day(), model.Time.Hour(), model.Time.Minute(), 0, 0, time.UTC)

		//model.Time
		itemAttributes := map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		}

		if model.Game != "" {
			itemAttributes["Game"] = &dynamodb.AttributeValue{S: aws.String(model.Game)}
		}
		if model.OldGame != "" {
			itemAttributes["OldGame"] = &dynamodb.AttributeValue{S: aws.String(model.OldGame)}
		}
		if model.GameID != "" {
			itemAttributes["GameID"] = &dynamodb.AttributeValue{S: aws.String(model.GameID)}
		}
		if model.OldGameID != "" {
			itemAttributes["OldGameID"] = &dynamodb.AttributeValue{S: aws.String(model.OldGameID)}
		}
		if model.Status != "" {
			itemAttributes["Status"] = &dynamodb.AttributeValue{S: aws.String(model.Status)}
		}
		if model.OldStatus != "" {
			itemAttributes["OldStatus"] = &dynamodb.AttributeValue{S: aws.String(model.OldStatus)}
		}
		if model.BroadcasterLanguage != "" {
			itemAttributes["BroadcasterLanguage"] = &dynamodb.AttributeValue{S: aws.String(model.BroadcasterLanguage)}
		}
		if model.OldBroadcasterLanguage != "" {
			itemAttributes["OldBroadcasterLanguage"] = &dynamodb.AttributeValue{S: aws.String(model.OldBroadcasterLanguage)}
		}

		input.RequestItems[TableChannelUpdates][i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: itemAttributes,
			},
		}
	}

	_, err := c.client.BatchWriteItemWithContext(ctx, input)

	return err
}

// GetAllByTime return ChannelUpdates given
// channelID and startTime and endTime.
// Returns []models.ChannelUpdate if found and error
// if something went wrong.
func (c *channelUpdateAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ChannelUpdate, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableChannelUpdates),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	result := make([]models.ChannelUpdate, *output.Count)
	for ind, value := range output.Items {
		model, err := c.buildModel(value)
		if err != nil {
			return nil, err
		}
		result[ind] = *model
	}

	return result, nil
}

// GetFirstBefore return last ChannelUpdates before given time.
// Returns models.ChannelUpdate if found and error
// if something went wrong.
func (c *channelUpdateAdapter) GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.ChannelUpdate, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T <= :time")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":time": {
			S: aws.String(timetime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:        aws.String(TableChannelUpdates),
		ScanIndexForward: aws.Bool(false),
		Limit:            aws.Int64(1),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	item := output.Items[0]
	model, err := c.buildModel(item)
	if err != nil {
		return nil, err
	}

	return model, nil
}

func (c *channelUpdateAdapter) buildModel(value map[string]*dynamodb.AttributeValue) (*models.ChannelUpdate, error) {
	channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
	if err != nil {
		return nil, err
	}

	var game, oldGame, gameID, oldGameID, status, oldStatus, broadcasterLanguage, oldBroadcasterLanguage string

	if value["Game"] != nil {
		game = *value["Game"].S
	}
	if value["OldGame"] != nil {
		oldGame = *value["OldGame"].S
	}
	if value["GameID"] != nil {
		gameID = *value["GameID"].S
	}
	if value["OldGameID"] != nil {
		oldGameID = *value["OldGameID"].S
	}
	if value["Status"] != nil {
		status = *value["Status"].S
	}
	if value["OldStatus"] != nil {
		oldStatus = *value["OldStatus"].S
	}
	if value["BroadcasterLanguage"] != nil {
		broadcasterLanguage = *value["BroadcasterLanguage"].S
	}
	if value["OldBroadcasterLanguage"] != nil {
		oldBroadcasterLanguage = *value["OldBroadcasterLanguage"].S
	}

	return &models.ChannelUpdate{
		ChannelID:              channelID,
		Time:                   timetime,
		Game:                   game,
		OldGame:                oldGame,
		GameID:                 gameID,
		OldGameID:              oldGameID,
		Status:                 status,
		OldStatus:              oldStatus,
		BroadcasterLanguage:    broadcasterLanguage,
		OldBroadcasterLanguage: oldBroadcasterLanguage,
	}, nil
}
