package adapters

import (
	"context"
	"fmt"
	"math"
	"sort"
	"strconv"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/go-redis/redis"

	"golang.org/x/sync/errgroup"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableChatActivity    = "CbChatActivity"
	CacheParticipantsKey = "participants"
	CacheMessagesKey     = "messages"
)

// ChatAdapter allows the consumer to save individual events into dynamodb
// and query an array of chat activity
type ChatAdapter interface {
	BatchSaveChatEvents(ctx context.Context, models []models.ChatEvent) error
	GetChatActivityByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) (models.ChatTimeSlice, error)
	GetChatActivityBySessions(ctx context.Context, channelID int64, sessionTimes []models.SessionTime) ([]models.ChatSummary, error)
}

type chatAdapter struct {
	client         dynamodbiface.DynamoDBAPI
	redisClient    *redis.Client
	sessionAdapter ChannelSessionAdapter
}

// NewChatAdapter create new processor
func NewChatAdapter(env string, region string, redisHost string) ChatAdapter {
	creds := helper.NewCredentials(env, region)

	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	client := redis.NewClient(&redis.Options{
		Addr:     redisHost,
		Password: "",
		DB:       0,
	})

	sessAdapter := NewChannelSessionAdapter(env, region)

	return &chatAdapter{
		client:         dynamodb.New(session.New(awsConfig)),
		redisClient:    client,
		sessionAdapter: sessAdapter,
	}
}

// BatchSaveChatEvents - saves an array of ChatEvent models. The function will
// attempt to condense ChatEvent models that have the same primary keys in an
// effort to reduce the number of UpdateItemWithContext() calls.
func (c *chatAdapter) BatchSaveChatEvents(ctx context.Context, chatEvents []models.ChatEvent) error {
	if len(chatEvents) == 0 {
		return nil
	}

	batchChatEvents := make(map[models.ChatEvent]int64)

	for _, chatEvent := range chatEvents {
		minute := int(math.Floor(float64(chatEvent.Time.Minute()/5)) * 5)
		flatTime := time.Date(chatEvent.Time.Year(), chatEvent.Time.Month(), chatEvent.Time.Day(), chatEvent.Time.Hour(), minute, 0, 0, time.UTC)
		chatEvent.Time = flatTime

		if _, ok := batchChatEvents[chatEvent]; ok {
			batchChatEvents[chatEvent]++
		} else {
			batchChatEvents[chatEvent] = 1
		}
	}

	for chatEvent, count := range batchChatEvents {
		emptyMap := make(map[string]*dynamodb.AttributeValue)
		emptyMap[strconv.FormatInt(chatEvent.UserID, 10)] = &dynamodb.AttributeValue{
			N: aws.String(strconv.FormatInt(count, 10)),
		}

		update := &dynamodb.UpdateItemInput{
			ExpressionAttributeNames: map[string]*string{
				"#UserID": aws.String(strconv.FormatInt(chatEvent.UserID, 10)),
			},
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":inc": {
					N: aws.String(strconv.FormatInt(count, 10)),
				},
				":zero": {
					N: aws.String("0"),
				},
			},
			Key: map[string]*dynamodb.AttributeValue{
				"ChannelID": &dynamodb.AttributeValue{
					N: aws.String(strconv.FormatInt(chatEvent.ChannelID, 10)),
				},
				"Time": &dynamodb.AttributeValue{
					S: aws.String(chatEvent.Time.Format(utils.DbTimeFormat)),
				},
			},
			TableName:        aws.String(TableChatActivity),
			UpdateExpression: aws.String("SET Chat.#UserID = if_not_exists(Chat.#UserID, :zero) + :inc"),
		}

		insert := &dynamodb.UpdateItemInput{
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":map": {
					M: emptyMap,
				},
			},
			Key: map[string]*dynamodb.AttributeValue{
				"ChannelID": &dynamodb.AttributeValue{
					N: aws.String(strconv.FormatInt(chatEvent.ChannelID, 10)),
				},
				"Time": &dynamodb.AttributeValue{
					S: aws.String(chatEvent.Time.Format(utils.DbTimeFormat)),
				},
			},
			TableName:           aws.String(TableChatActivity),
			UpdateExpression:    aws.String("SET Chat = if_not_exists(Chat, :map)"),
			ConditionExpression: aws.String("attribute_not_exists(Chat)"),
		}

		query := update
		for {
			_, err := c.client.UpdateItemWithContext(ctx, query)
			if err != nil {
				if strings.Contains(err.Error(), PathError) {
					query = insert
				} else if strings.Contains(err.Error(), ConditionError) {
					query = update
				} else {
					return err
				}
			} else {
				break
			}
		}
	}

	return nil
}

// GetChatActivityByTime returns an array of ChatActivity structs for a given
// ChannelID between Start Time and End Time. Each record is a 5 minute chunk.
func (c *chatAdapter) GetChatActivityByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) (models.ChatTimeSlice, error) {
	activity, err := queryChatActivity(ctx, c.client, channelID, startTime, endTime)
	if err != nil {
		return nil, err
	}

	sort.Sort(models.ChatTimeSlice(activity))

	return activity, nil
}

func (c *chatAdapter) GetChatActivityBySessions(ctx context.Context, channelID int64, sessionTimes []models.SessionTime) ([]models.ChatSummary, error) {
	if len(sessionTimes) == 0 {
		return nil, nil
	}

	// We can use this slice to write the results directly to an array in each goroutine used below.
	// This saves us the need to define a sort on the final result, which would be required if
	// we used a channel. This is only possible because we know the number of goroutines ahead of time,
	// so we can be sure that the underlying array oof the slice never changes.
	chatSummaryResults := make([]models.ChatSummary, len(sessionTimes))
	group, ctx := errgroup.WithContext(ctx)

	for i, p := range sessionTimes {
		pair := p
		idx := i
		group.Go(func() error {
			// Check redis for a cached version of this session. If found, return it here.
			// Redis key is of the format: ChannelID_StartTime_EndTime
			redisKey := buildRedisKey(channelID, pair.StartTime, pair.EndTime)
			cachedSessionMap, err := c.redisClient.HGetAll(redisKey).Result()

			if err != nil {
				return err
			}

			if len(cachedSessionMap) > 0 {
				timestamp := models.DynamoDBTimestamp{
					Converted: pair.StartTime,
				}

				participants, err := strconv.ParseInt(cachedSessionMap[CacheParticipantsKey], 10, 64)
				if err != nil {
					return err
				}

				messages, err := strconv.ParseInt(cachedSessionMap[CacheMessagesKey], 10, 64)
				if err != nil {
					return err
				}

				sessionChatSummary := models.ChatSummary{
					Time:         timestamp,
					ChannelID:    channelID,
					Participants: participants,
					Messages:     messages,
				}

				chatSummaryResults[idx] = sessionChatSummary

				return nil
			}

			chatActivity, err := queryChatActivity(ctx, c.client, channelID, pair.StartTime, pair.EndTime)

			if err != nil {
				return err
			}

			var messagesTotal int64
			chattersMap := make(map[string]int64)
			for _, chatActivityTimeSlice := range chatActivity {
				for user, messages := range chatActivityTimeSlice.Chat {
					messagesTotal += messages
					if _, ok := chattersMap[user]; ok {
						chattersMap[user] += messages
					} else {
						chattersMap[user] = messages
					}
				}
			}

			redisMap := make(map[string]interface{})
			redisMap[CacheParticipantsKey] = int64(len(chattersMap))
			redisMap[CacheMessagesKey] = messagesTotal

			// Expiry time is 30 days. Since go doesn't have any units of
			// time over hours, we set 30 days to mean 720 hours.
			expiryDuration := time.Duration(720) * time.Hour

			c.redisClient.HMSet(redisKey, redisMap)
			c.redisClient.Expire(redisKey, expiryDuration)

			timestamp := models.DynamoDBTimestamp{
				Converted: pair.StartTime,
			}

			sessionChatSummary := models.ChatSummary{
				Time:         timestamp,
				ChannelID:    channelID,
				Participants: int64(len(chattersMap)),
				Messages:     messagesTotal,
			}

			chatSummaryResults[idx] = sessionChatSummary

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		return nil, err
	}

	return chatSummaryResults, nil
}

func queryChatActivity(ctx context.Context, client dynamodbiface.DynamoDBAPI, channelID int64, startTime time.Time, endTime time.Time) ([]models.ChatActivity, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")

	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableChatActivity),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	subResults := []models.ChatActivity{}
	err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &subResults)

	if err != nil {
		return nil, err
	}

	return subResults, nil
}

// Given some values present in chat-stats redis keys, builds a string to create the correct key
func buildRedisKey(channelID int64, startTime time.Time, endTime time.Time) string {
	return fmt.Sprintf("%d_%s_%s", channelID, startTime.String(), endTime.String())
}
