package adapters

import (
	"context"
	"math"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableClipIDToChannelTime = "CbClipIDToChannelTime"
	TableClips               = "CbClips"

	RedditDomain   = "reddit"
	OtherDomain    = "other"
	FacebookDomain = "facebook"
	TwitterDomain  = "twitter"
	TwitchDomain   = "twitch"

	PlaysColumn         = "Plays"
	RedditPlaysColumn   = "RedditPlays"
	OtherPlaysColumn    = "OtherPlays"
	FacebookPlaysColumn = "FacebookPlays"
	TwitterPlaysColumn  = "TwitterPlays"
	TwitchPlaysColumn   = "TwitchPlays"
)

//
// ClipsAdapter processor.
//
type ClipsAdapter interface {
	UpdateViewCount(ctx context.Context, model models.Clip) error
	UpdateCreateCount(ctx context.Context, model models.Clip) error
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.Clip, error)
}

type clipsAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewClipsAdapter create new processor.
func NewClipsAdapter(env string, region string) ClipsAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		Credentials: creds,
		Region:      aws.String(region),
	}

	return &clipsAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// Save - saves model to dynamodb table.
func (c *clipsAdapter) UpdateViewCount(ctx context.Context, model models.Clip) error {
	return c.upsertClipVideoPlay(ctx, model)
}

func (c *clipsAdapter) UpdateCreateCount(ctx context.Context, model models.Clip) error {
	return c.upsertClipCreate(ctx, model)
}

// upsertClipCreate - creates the clip create record if it doesn't exist, increments it otherwise
func (c *clipsAdapter) upsertClipCreate(ctx context.Context, obj models.Clip) error {
	minute := int(math.Floor(float64(obj.Time.Minute()/5)) * 5)
	flatTime := time.Date(obj.Time.Year(), obj.Time.Month(),
		obj.Time.Day(), obj.Time.Hour(), minute, 0, 0, time.UTC)

	_, err := c.client.UpdateItemWithContext(ctx, &dynamodb.UpdateItemInput{
		UpdateExpression: aws.String("ADD Creates :IncBy"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":IncBy": {N: aws.String(strconv.FormatInt(obj.Creates, 10))},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": {N: aws.String(strconv.FormatInt(obj.ChannelID, 10))},
			"Time":      {S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName: aws.String(TableClips),
	})

	if err != nil {
		return err
	}

	_, err = c.client.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"ClipID": &dynamodb.AttributeValue{S: aws.String(obj.ClipID)},
			"Time":   &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName: aws.String(TableClipIDToChannelTime),
	})

	return err
}

// upsertClipVideoPlay - creates the video play record if it doesn't exist, increments it otherwise
func (c *clipsAdapter) upsertClipVideoPlay(ctx context.Context, obj models.Clip) error {
	// Fetch the time the clip was made
	clipTime, err := c.getClipTime(ctx, obj.ClipID)
	if err != nil {
		return err
	}

	// We don't have this clip creation logged, skip handling of the play event
	if clipTime.IsZero() {
		return nil
	}

	_, err = c.client.UpdateItemWithContext(ctx, &dynamodb.UpdateItemInput{
		UpdateExpression: aws.String(
			"ADD #PlaysColumn :PlaysIncBy, #TwitchPlaysColumn :TwitchIncBy, #FacebookPlaysColumn :FacebookIncBy, #TwitterPlaysColumn :TwitterIncBy, #RedditPlaysColumn :RedditIncBy, #OtherPlaysColumn :OtherIncBy",
		),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":PlaysIncBy":    {N: aws.String(strconv.FormatInt(obj.Plays, 10))},
			":TwitchIncBy":   {N: aws.String(strconv.FormatInt(obj.PlaysBreakdown[TwitchDomain], 10))},
			":FacebookIncBy": {N: aws.String(strconv.FormatInt(obj.PlaysBreakdown[FacebookDomain], 10))},
			":TwitterIncBy":  {N: aws.String(strconv.FormatInt(obj.PlaysBreakdown[TwitterDomain], 10))},
			":RedditIncBy":   {N: aws.String(strconv.FormatInt(obj.PlaysBreakdown[RedditDomain], 10))},
			":OtherIncBy":    {N: aws.String(strconv.FormatInt(obj.PlaysBreakdown[OtherDomain], 10))},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": {N: aws.String(strconv.FormatInt(obj.ChannelID, 10))},
			"Time":      {S: aws.String(clipTime.Format(utils.DbTimeFormat))},
		},
		ExpressionAttributeNames: map[string]*string{
			"#PlaysColumn":         aws.String(PlaysColumn),
			"#TwitchPlaysColumn":   aws.String(TwitchPlaysColumn),
			"#FacebookPlaysColumn": aws.String(FacebookPlaysColumn),
			"#TwitterPlaysColumn":  aws.String(TwitterPlaysColumn),
			"#RedditPlaysColumn":   aws.String(RedditPlaysColumn),
			"#OtherPlaysColumn":    aws.String(OtherPlaysColumn),
		},
		TableName: aws.String(TableClips),
	})

	return err
}

func (c *clipsAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.Clip, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableClips),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	result := make([]models.Clip, *output.Count)
	for ind, value := range output.Items {
		timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
		if err != nil {
			return nil, err
		}

		plays, err := utils.MaybeInt64(value, PlaysColumn)
		if err != nil {
			return nil, err
		}

		redditPlays, err := utils.MaybeInt64(value, RedditPlaysColumn)
		if err != nil {
			return nil, err
		}

		otherPlays, err := utils.MaybeInt64(value, OtherPlaysColumn)
		if err != nil {
			return nil, err
		}

		facebookPlays, err := utils.MaybeInt64(value, FacebookPlaysColumn)
		if err != nil {
			return nil, err
		}

		twitterPlays, err := utils.MaybeInt64(value, TwitterPlaysColumn)
		if err != nil {
			return nil, err
		}

		twitchPlays, err := utils.MaybeInt64(value, TwitchPlaysColumn)
		if err != nil {
			return nil, err
		}

		playsBreakdown := map[string]int64{
			RedditDomain:   redditPlays,
			OtherDomain:    otherPlays,
			FacebookDomain: facebookPlays,
			TwitterDomain:  twitterPlays,
			TwitchDomain:   twitchPlays,
		}

		creates, err := utils.MaybeInt64(value, "Creates")
		if err != nil {
			return nil, err
		}

		result[ind] = models.Clip{
			ChannelID:      channelID,
			Time:           timetime,
			Plays:          plays,
			Creates:        creates,
			PlaysBreakdown: playsBreakdown,
		}
	}

	return result, nil
}

func (c *clipsAdapter) getClipTime(ctx context.Context, clipID string) (time.Time, error) {
	keyCondition := aws.String("ClipID = :ClipID")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":ClipID": {S: aws.String(clipID)},
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableClipIDToChannelTime),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
	})

	if err != nil {
		return time.Time{}, err
	}

	if *output.Count == 0 {
		return time.Time{}, nil
	}

	item := output.Items[0]
	timetime, err := time.Parse(utils.DbTimeFormat, *item["Time"].S)
	if err != nil {
		return time.Time{}, err
	}

	return timetime, nil
}
