package adapters

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableCommercials = "CbCommercials"
)

type CommercialAdapter interface {
	SaveCommercial(ctx context.Context, model models.Commercial) error
	GetCommercialsByTime(ctx context.Context,
		channelID int64,
		startTime time.Time,
		endTime time.Time) ([]models.Commercial, error)
}

type commercialAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

func NewCommercialAdapter(env string, region string) CommercialAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &commercialAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

func (c *commercialAdapter) SaveCommercial(ctx context.Context, model models.Commercial) error {
	if model.ChannelID == 0 {
		return nil
	}

	_, err := c.client.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(model.Time.Converted.Format(utils.DbTimeFormat))},
			"Length":    &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.Length, 10))},
		},
		TableName: aws.String(TableCommercials),
	})

	return err
}

func (c *commercialAdapter) GetCommercialsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.Commercial, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")

	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}

	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableCommercials),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *result.Count == 0 {
		return nil, nil
	}

	commercials := []models.Commercial{}
	err = dynamodbattribute.UnmarshalListOfMaps(result.Items, &commercials)

	return commercials, err
}
