package adapters

import (
	"context"
	"math"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableFollowsAggregate = "CbFollowsAggregate"
)

//
// FollowAggregateUpdateAdapter processor.
//
type FollowAggregateUpdateAdapter interface {
	// BatchSave - saves model into DynamoDatabase under specific key
	// defined by channel_id and time defined in model.
	BatchSave(ctx context.Context, models []models.FollowAggregateUpdate) error

	// GetAllByTime return ChannelUpdates given
	// channelID and startTime and endTime.
	// Returns []models.ChannelUpdate if found and error
	// if something went wrong.
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.FollowAggregateUpdate, error)

	// GetFirstBefore return last ChannelUpdates before given time.
	// Returns models.ChannelUpdate if found and error
	// if something went wrong.
	GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.FollowAggregateUpdate, error)
}

type followAggregateUpdateAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewFollowAggregateUpdateAdapter create new processor.
func NewFollowAggregateUpdateAdapter(env string, region string) FollowAggregateUpdateAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &followAggregateUpdateAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// BatchSave - saves model into DynamoDatabase under specific key
// defined by channel_id and time.
func (c *followAggregateUpdateAdapter) BatchSave(ctx context.Context, models []models.FollowAggregateUpdate) error {
	if len(models) == 0 {
		return nil
	}

	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			TableFollowsAggregate: make([]*dynamodb.WriteRequest, len(models)),
		},
		ReturnConsumedCapacity: aws.String(dynamodb.ReturnConsumedCapacityTotal),
	}

	for i, model := range models {
		minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
		flatTime := time.Date(model.Time.Year(), model.Time.Month(),
			model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

		//model.Time
		itemAttributes := map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Total":     &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.Total, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		}

		input.RequestItems[TableFollowsAggregate][i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: itemAttributes,
			},
		}
	}

	_, err := c.client.BatchWriteItemWithContext(ctx, input)

	return err
}

// GetAllByTime return all FollowAggregateUpdate events given
// channelID and startTime and endTime.
// Returns []models.FollowAggregateUpdate if found and error
// if something went wrong.
func (c *followAggregateUpdateAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.FollowAggregateUpdate, error) {
	// this is for pagination
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.FollowAggregateUpdate{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableFollowsAggregate),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		for _, value := range output.Items {
			model, err := c.buildModel(value)
			if err != nil {
				return nil, err
			}

			result = append(result, *model)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetFirstBefore return last FollowAggregateUpdate event before given time.
// Returns models.FollowAggregateUpdate if found and error
// if something went wrong.
func (c *followAggregateUpdateAdapter) GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.FollowAggregateUpdate, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T <= :time")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":time": {
			S: aws.String(timetime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:        aws.String(TableFollowsAggregate),
		ScanIndexForward: aws.Bool(false),
		Limit:            aws.Int64(1),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	item := output.Items[0]
	model, err := c.buildModel(item)
	if err != nil {
		return nil, err
	}

	return model, nil
}

func (c *followAggregateUpdateAdapter) buildModel(value map[string]*dynamodb.AttributeValue) (*models.FollowAggregateUpdate, error) {
	channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
	if err != nil {
		return nil, err
	}

	var total int64
	if value["Total"] != nil {
		if parsed, err := strconv.ParseInt(*value["Total"].N, 10, 64); err == nil {
			total = parsed
		}
	}

	return &models.FollowAggregateUpdate{
		ChannelID: channelID,
		Time:      timetime,
		Total:     total,
	}, nil
}
