package adapters

import (
	"context"
	"math"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableIncrementalFollow = "CbIncrementalFollows"
)

//
// IncrementalFollowAdapter processor.
//
type IncrementalFollowAdapter interface {
	// Save - increment the counter for the incremental follow table
	// under the specified key. AWS SDK go doesnt support batch updates
	// biblethump
	Save(ctx context.Context, models models.IncrementalFollow) error

	// GetAllByTime return all IncrementalFollow events
	// provided a channelID and a time range.
	// Returns []models.IncrementalFollow if found and error
	// if something went wrong.
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.IncrementalFollow, error)

	// GetFirstBefore return last IncrementalFollow event before given time.
	// Returns models.IncrementalFollow if found and error
	// if something went wrong.
	GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.IncrementalFollow, error)
}

type incrementalFollowAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewIncrementalFollowAdapter create new processor.
func NewIncrementalFollowAdapter(env string, region string) IncrementalFollowAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &incrementalFollowAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// Save - increment the counter for the incremental follow table
// under the specified key. AWS SDK go doesnt support batch updates
// biblethump
func (c *incrementalFollowAdapter) Save(ctx context.Context, model models.IncrementalFollow) error {
	minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	update := &dynamodb.UpdateItemInput{
		ExpressionAttributeNames: map[string]*string{
			"#C": aws.String("Count"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":i": {
				N: aws.String("1"),
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:        aws.String(TableIncrementalFollow),
		UpdateExpression: aws.String("ADD #C :i"),
	}

	_, err := c.client.UpdateItemWithContext(ctx, update)
	return err
}

// GetAllByTime return all IncrementalFollow events given
// channelID and startTime and endTime.
// Returns []models.IncrementalFollow if found and error
// if something went wrong.
func (c *incrementalFollowAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.IncrementalFollow, error) {
	// this is for pagination
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.IncrementalFollow{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableIncrementalFollow),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		for _, value := range output.Items {
			model, err := c.buildModel(value)
			if err != nil {
				return nil, err
			}

			result = append(result, *model)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetFirstBefore return last IncrementalFollow event before given time.
// Returns models.IncrementalFollow if found and error
// if something went wrong.
func (c *incrementalFollowAdapter) GetFirstBefore(ctx context.Context, channelID int64, timetime time.Time) (*models.IncrementalFollow, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T <= :time")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":time": {
			S: aws.String(timetime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:        aws.String(TableIncrementalFollow),
		ScanIndexForward: aws.Bool(false),
		Limit:            aws.Int64(1),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	item := output.Items[0]
	model, err := c.buildModel(item)
	if err != nil {
		return nil, err
	}

	return model, nil
}

func (c *incrementalFollowAdapter) buildModel(value map[string]*dynamodb.AttributeValue) (*models.IncrementalFollow, error) {
	channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
	if err != nil {
		return nil, err
	}

	var count int64
	if value["Count"] != nil {
		if parsed, err := strconv.ParseInt(*value["Count"].N, 10, 64); err == nil {
			count = parsed
		}
	}

	return &models.IncrementalFollow{
		ChannelID: channelID,
		Count:     count,
		Time:      timetime,
	}, nil
}
