package adapters

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableMinuteBroadcasts = "CbMinuteBroadcasts"

	// in minutes
	SessionMinuteGap = 15

	// in hours
	SessionLengthCutoff = 48
	SessionLengthMax    = 24
)

//
// MinuteBroadcastAdapter processor.
//
type MinuteBroadcastAdapter interface {
	// BatchSave - saves model into DynamoDatabase under specific key
	// defined by channel_id and time defined in model.
	BatchSave(ctx context.Context, models []models.MinuteBroadcast) error

	// GetAllByTime return MinuteBroadcasts given
	// channelID and startTime and endTime.
	// Returns []models.MinuteBroadcast if found and error
	// if something went wrong.
	GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.MinuteBroadcast, error)

	// GetBeforeTime returns one MinuteBroadcast immediately
	// before t (time.Time) given channelID.
	//
	// Returns models.MinuteBroadcast if found and error
	// if something went wrong.
	GetBeforeTime(ctx context.Context, channelID int64, t time.Time) (*models.MinuteBroadcast, error)

	// CalculateLastSessions return MinuteBroadcasts given
	// channelID and size of list.
	// Returns []models.ChannelSession on success and error
	// if something went wrong.
	CalculateLastSessions(ctx context.Context, channelID int64, size int) ([]models.ChannelSession, error)
}

type minuteBroadcastAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewMinuteBroadcastAdapter create new processor.
func NewMinuteBroadcastAdapter(env string, region string) MinuteBroadcastAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &minuteBroadcastAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// BatchSave - saves model into DynamoDatabase under specific key
// defined by channel_id and time.
func (c *minuteBroadcastAdapter) BatchSave(ctx context.Context, models []models.MinuteBroadcast) error {
	if len(models) == 0 {
		return nil
	}

	input := &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			TableMinuteBroadcasts: make([]*dynamodb.WriteRequest, len(models)),
		},
		ReturnConsumedCapacity: aws.String(dynamodb.ReturnConsumedCapacityTotal),
	}

	for i, model := range models {
		flatTime := time.Date(model.Time.Year(), model.Time.Month(),
			model.Time.Day(), model.Time.Hour(), model.Time.Minute(), 0, 0, time.UTC)

		//model.Time
		itemAttributes := map[string]*dynamodb.AttributeValue{
			"ChannelID":   &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":        &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
			"BroadcastID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.BroadcastID, 10))},
		}

		if model.Game != "" {
			itemAttributes["Game"] = &dynamodb.AttributeValue{S: aws.String(model.Game)}
		}

		if model.BroadcasterSoftware != "" {
			itemAttributes["BroadcasterSoftware"] = &dynamodb.AttributeValue{S: aws.String(model.BroadcasterSoftware)}
		}

		input.RequestItems[TableMinuteBroadcasts][i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: itemAttributes,
			},
		}
	}

	_, err := c.client.BatchWriteItemWithContext(ctx, input)

	return err
}

// GetAllByTime return MinuteBroadcasts given
// channelID and startTime and endTime.
// Returns []models.MinuteBroadcast if found and error
// if something went wrong.
func (c *minuteBroadcastAdapter) GetAllByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.MinuteBroadcast, error) {
	// this is for pagination
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.MinuteBroadcast{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableMinuteBroadcasts),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		for _, value := range output.Items {
			model, err := c.buildModel(value)
			if err != nil {
				return nil, err
			}

			result = append(result, *model)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetBeforeTime returns one MinuteBroadcast immediately
// before t (time.Time) given channelID.
//
// Returns models.MinuteBroadcast if found and error
// if something went wrong.
func (c *minuteBroadcastAdapter) GetBeforeTime(ctx context.Context, channelID int64, t time.Time) (*models.MinuteBroadcast, error) {
	keyCondition := aws.String("ChannelID = :channelID AND #T < :time")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":time": {
			S: aws.String(t.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		ExpressionAttributeNames:  attributePlaceholders,
		ExpressionAttributeValues: conditionAttrValues,
		KeyConditionExpression:    keyCondition,
		Limit:            aws.Int64(1),
		ScanIndexForward: aws.Bool(false),
		TableName:        aws.String(TableMinuteBroadcasts),
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	return c.buildModel(output.Items[0])
}

// CalculateLastSessions return MinuteBroadcasts given
// channelID and size of list.
// Returns []models.ChannelSession on success and error
// if something went wrong.
func (c *minuteBroadcastAdapter) CalculateLastSessions(ctx context.Context, channelID int64, size int) ([]models.ChannelSession, error) {
	var exclusiveStartKey map[string]*dynamodb.AttributeValue

	// Sessions
	sessions := []models.ChannelSession{}
	keyCondition := aws.String("ChannelID = :channelID")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
	}

	// Infinite cycle till end of records or size satisfied
	var currentSession *models.ChannelSession

	broadcastIDs := [][]int64{}
	partitionedSession := false

	for {

		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:        aws.String(TableMinuteBroadcasts),
			ScanIndexForward: aws.Bool(false),
			Limit:            aws.Int64(1000),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		if *output.Count == 0 {
			break
		}

		for _, value := range output.Items {
			// Return sessions if reach maximum required sessions.
			if len(sessions) >= size {
				return sessions, nil
			}

			channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
			if err != nil {
				return nil, err
			}

			timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
			if err != nil {
				return nil, err
			}
			broadcastID, err := strconv.ParseInt(*value["BroadcastID"].N, 10, 64)
			if err != nil {
				return nil, err
			}

			// No session, so create a new one
			if currentSession == nil {
				currentSession = &models.ChannelSession{
					ChannelID: channelID,
					StartTime: timetime,
					EndTime:   timetime,
				}
				broadcastIDs = [][]int64{{broadcastID}}
				continue
			}

			// Regular session that is identified by gap between minute broadcast.
			if currentSession.StartTime.Sub(timetime) >= SessionMinuteGap*time.Minute {
				currentSession.BroadcastIDs = utils.FlattenInt64List(broadcastIDs)
				sessions = append(sessions, *currentSession)

				// reset session
				currentSession = &models.ChannelSession{
					ChannelID: channelID,
					StartTime: timetime,
					EndTime:   timetime,
				}
				broadcastIDs = [][]int64{{broadcastID}}
				// reset partition session
				partitionedSession = false
				continue
			}

			// Update current session with new time and broadcastID.
			currentSession.StartTime = timetime
			broadcastIDs[len(broadcastIDs)-1] = utils.AppendUniqueInt64(broadcastIDs[len(broadcastIDs)-1], broadcastID)

			// Check if current session is getting out of SessionLengthCutoff and trigger partitionedSession
			if currentSession.EndTime.Sub(currentSession.StartTime) >= SessionLengthCutoff*time.Hour {
				partitionedSession = true
				// ......currentSession[<-----(2)24h----><-----(1)24h---->]
				// broadcastIDs = [ [(1) ids], [(2) ids ]
				// Current session is 48h long, we need to split it in 24h chunks and store them.
				// All next sessions will be stored later down the code.
				session1 := &models.ChannelSession{
					ChannelID:    currentSession.ChannelID,
					StartTime:    currentSession.EndTime.Add(-1 * time.Duration(SessionLengthMax) * time.Hour),
					EndTime:      currentSession.EndTime,
					BroadcastIDs: broadcastIDs[0],
				}

				session2 := &models.ChannelSession{
					ChannelID:    currentSession.ChannelID,
					StartTime:    currentSession.StartTime,
					EndTime:      currentSession.EndTime.Add(-1 * time.Duration(SessionLengthMax) * time.Hour),
					BroadcastIDs: broadcastIDs[1],
				}

				sessions = append(sessions, *session1, *session2)
				// Reset by setting currentSession nil
				currentSession = nil
				continue
			}

			// If session is longer than 24h we are preparing for 24/7 session.
			if currentSession.EndTime.Sub(currentSession.StartTime) >= SessionLengthMax*time.Hour {
				broadcastIDs = append(broadcastIDs, []int64{broadcastID})

				// Check if we are in partition mode and each 24h should be stored individually
				if partitionedSession {
					currentSession.BroadcastIDs = utils.FlattenInt64List(broadcastIDs)
					sessions = append(sessions, *currentSession)

					// reset session
					currentSession = nil
					continue
				}
			}
		}

		// Return sessions if reach maximum required sessions.
		if len(sessions) >= size {
			return sessions, nil
		}

		// No more records
		if output.LastEvaluatedKey == nil {
			break
		}
		exclusiveStartKey = output.LastEvaluatedKey
	}

	if currentSession != nil {
		sessions = append(sessions, *currentSession)
	}

	return sessions, nil
}

func (c *minuteBroadcastAdapter) buildModel(value map[string]*dynamodb.AttributeValue) (*models.MinuteBroadcast, error) {
	channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
	if err != nil {
		return nil, err
	}

	broadcastID, err := strconv.ParseInt(*value["BroadcastID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	var game string
	if value["Game"] != nil {
		game = *value["Game"].S
	}

	var software string
	if value["BroadcasterSoftware"] != nil {
		software = *value["BroadcasterSoftware"].S
	}

	return &models.MinuteBroadcast{
		ChannelID:           channelID,
		Time:                timetime,
		Game:                game,
		BroadcastID:         broadcastID,
		BroadcasterSoftware: software,
	}, nil
}
