package adapters

import (
	"context"
	"strconv"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	TableRaids = "CbRaids"
)

//
// RaidsAdapter processor.
//
type RaidsAdapter interface {
	SaveRaid(ctx context.Context, model models.Raid) error
	GetAllByTime(ctx context.Context, targetID int64, startTime time.Time, endTime time.Time) ([]models.Raid, error)
}

type raidsAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewRaidsAdapter create new processor.
func NewRaidsAdapter(env string, region string) RaidsAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		Credentials: creds,
		Region:      aws.String(region),
	}

	return &raidsAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

func (c *raidsAdapter) SaveRaid(ctx context.Context, obj models.Raid) error {
	if obj.RaidID == "" || obj.TargetID == "" {
		return nil
	}

	_, err := c.client.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"RaidID":      &dynamodb.AttributeValue{S: aws.String(obj.RaidID)},
			"TargetID":    &dynamodb.AttributeValue{S: aws.String(obj.TargetID)},
			"Time":        &dynamodb.AttributeValue{S: aws.String(obj.Time.Converted.Format(utils.DbTimeFormat))},
			"ViewerCount": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(obj.ViewerCount, 10))},
		},
		TableName: aws.String(TableRaids),
	})

	return err
}

func (c *raidsAdapter) GetAllByTime(ctx context.Context, targetID int64, startTime time.Time, endTime time.Time) ([]models.Raid, error) {
	keyCondition := aws.String("TargetID = :targetID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":targetID": {
			S: aws.String(strconv.FormatInt(targetID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableRaids),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
		ExpressionAttributeNames:  attributePlaceholders,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	result := []models.Raid{}
	err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &result)

	return result, err
}
