package adapters

import (
	"context"
	"math"
	"strconv"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
)

const (
	TableChannelGeo         = "CbChannelGeo"
	TableChannelReferral    = "CbChannelReferral"
	TableChannelPlatform    = "CbChannelPlatform"
	TableChannelUniquePlays = "CbChannelUniquePlays"

	PathError      = "The document path provided in the update expression is invalid for update"
	ConditionError = "The conditional request failed"
)

var EmberPrimaryDirectories = map[string]bool{
	"videos":             true,
	"collections":        true,
	"inventory":          true,
	"bits":               true,
	"404":                true,
	"pr":                 true,
	"settings":           true,
	"private":            true,
	"create-landing":     true,
	"communities":        true,
	"friends":            true,
	"extensions":         true,
	"events":             true,
	"streamplus":         true,
	"streams":            true,
	"youtubeRedirect":    true,
	"broadcast":          true,
	"backpack":           true,
	"email-verification": true,
	"directory":          true,
	"index":              true,
	"team":               true,
}

//
// VideoPlayAdapter processor.
//
type VideoPlayAdapter interface {
	//for each of these tables we classify the video-play event and save it
	SaveGeoData(ctx context.Context, models models.VideoPlay) error
	SaveReferralData(ctx context.Context, models models.VideoPlay) error
	SavePlatformData(ctx context.Context, models models.VideoPlay) error
	SaveDevicesData(ctx context.Context, models models.VideoPlay) error

	//query bucket aggregates for each table
	GetGeoMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.GeoMap, error)
	GetUniqueViewsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.UniqueDevices, error)
	GetReferralMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ReferralMap, error)
	GetPlatformMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.PlatformMap, error)
}

type videoPlayAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewChannelConcurrentAdapter create new processor.
func NewVideoPlayAdapter(env string, region string) VideoPlayAdapter {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      creds,
		Region:           aws.String(region),
	}

	return &videoPlayAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}
}

// SaveGeoData - bucket sort video-play events by the mindmax country
// 	"unknown" if country is null
func (c *videoPlayAdapter) SaveGeoData(ctx context.Context, model models.VideoPlay) error {
	minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	country := model.Country
	if len(country) == 0 {
		country = "unknown"
	}
	emptyMap := make(map[string]*dynamodb.AttributeValue)
	emptyMap[country] = &dynamodb.AttributeValue{
		N: aws.String("1"),
	}

	update := &dynamodb.UpdateItemInput{
		ExpressionAttributeNames: map[string]*string{
			"#Country": aws.String(country),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":inc": {
				N: aws.String("1"),
			},
			":zero": {
				N: aws.String("0"),
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:        aws.String(TableChannelGeo),
		UpdateExpression: aws.String("SET Geo.#Country = if_not_exists(Geo.#Country, :zero) + :inc"),
	}

	insert := &dynamodb.UpdateItemInput{
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":map": {
				M: emptyMap,
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:           aws.String(TableChannelGeo),
		UpdateExpression:    aws.String("SET Geo = if_not_exists(Geo, :map)"),
		ConditionExpression: aws.String("attribute_not_exists(Geo)"),
	}

	query := update
	for {
		_, err := c.client.UpdateItemWithContext(ctx, query)
		if err != nil {
			if strings.Contains(err.Error(), PathError) {
				query = insert
			} else if strings.Contains(err.Error(), ConditionError) {
				query = update
			} else {
				return err
			}
		} else {
			return nil
		}
	}
}

// SaveReferralData - bucket sort video-play events by internal v external
//	location and additional metadata
func (c *videoPlayAdapter) SaveReferralData(ctx context.Context, model models.VideoPlay) error {
	var refSource string
	minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	refType := getReferrerSourceTypeBucket(&model)
	if refType == "Internal" {
		refSource = getInternalReferrerSourceBucket(&model)
	} else {
		refSource = getExternalReferrerSourceBucket(&model)
	}

	emptyMap := make(map[string]*dynamodb.AttributeValue)
	emptyMap[refSource] = &dynamodb.AttributeValue{
		N: aws.String("1"),
	}

	update := &dynamodb.UpdateItemInput{
		ExpressionAttributeNames: map[string]*string{
			"#Type":   aws.String(refType),
			"#Source": aws.String(refSource),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":inc": {
				N: aws.String("1"),
			},
			":zero": {
				N: aws.String("0"),
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:        aws.String(TableChannelReferral),
		UpdateExpression: aws.String("SET #Type.#Source = if_not_exists(#Type.#Source, :zero) + :inc"),
	}

	insert := &dynamodb.UpdateItemInput{
		ExpressionAttributeNames: map[string]*string{
			"#Type": aws.String(refType),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":map": {
				M: emptyMap,
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:           aws.String(TableChannelReferral),
		UpdateExpression:    aws.String("SET #Type = if_not_exists(#Type, :map)"),
		ConditionExpression: aws.String("attribute_not_exists(#Type)"),
	}

	query := update
	for {
		_, err := c.client.UpdateItemWithContext(ctx, query)
		if err != nil {
			if strings.Contains(err.Error(), PathError) {
				query = insert
			} else if strings.Contains(err.Error(), ConditionError) {
				query = update
			} else {
				return err
			}
		} else {
			return nil
		}
	}
}

// SavePlatformData - bucket sort video-play events by platform and device
func (c *videoPlayAdapter) SavePlatformData(ctx context.Context, model models.VideoPlay) error {
	minute := int(math.Floor(float64(model.Time.Minute()/5)) * 5)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	bucket := getPlatformBucket(model.Platform, model.Player)
	emptyMap := make(map[string]*dynamodb.AttributeValue)
	emptyMap[bucket] = &dynamodb.AttributeValue{
		N: aws.String("1"),
	}

	update := &dynamodb.UpdateItemInput{
		ExpressionAttributeNames: map[string]*string{
			"#Platform": aws.String(bucket),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":inc": {
				N: aws.String("1"),
			},
			":zero": {
				N: aws.String("0"),
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:        aws.String(TableChannelPlatform),
		UpdateExpression: aws.String("SET Platform.#Platform = if_not_exists(Platform.#Platform, :zero) + :inc"),
	}

	insert := &dynamodb.UpdateItemInput{
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":map": {
				M: emptyMap,
			},
		},
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
			"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
		},
		TableName:           aws.String(TableChannelPlatform),
		UpdateExpression:    aws.String("SET Platform = if_not_exists(Platform, :map)"),
		ConditionExpression: aws.String("attribute_not_exists(Platform)"),
	}

	query := update
	for {
		_, err := c.client.UpdateItemWithContext(ctx, query)
		if err != nil {
			if strings.Contains(err.Error(), PathError) {
				query = insert
			} else if strings.Contains(err.Error(), ConditionError) {
				query = update
			} else {
				return err
			}
		} else {
			return nil
		}
	}
}

// BatchSave - saves unique into DynamoDB under specific key using channelId
// defined by channel_id and time, device_ids defined in model.
func (c *videoPlayAdapter) SaveDevicesData(ctx context.Context, model models.VideoPlay) error {
	minute := int(math.Floor(float64(model.Time.Minute()/1)) * 1)
	flatTime := time.Date(model.Time.Year(), model.Time.Month(),
		model.Time.Day(), model.Time.Hour(), minute, 0, 0, time.UTC)

	deviceId := model.DeviceID
	if len(deviceId) > 0 {
		emptyMap := make(map[string]*dynamodb.AttributeValue)
		emptyMap[deviceId] = &dynamodb.AttributeValue{
			N: aws.String("1"),
		}

		update := &dynamodb.UpdateItemInput{
			ExpressionAttributeNames: map[string]*string{
				"#DeviceID": aws.String(deviceId),
			},
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":inc": {
					N: aws.String("1"),
				},
				":zero": {
					N: aws.String("0"),
				},
			},
			Key: map[string]*dynamodb.AttributeValue{
				"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
				"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
			},
			TableName:        aws.String(TableChannelUniquePlays),
			UpdateExpression: aws.String("SET DeviceID.#DeviceID = if_not_exists(DeviceID.#DeviceID, :zero) + :inc"),
		}

		insert := &dynamodb.UpdateItemInput{
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":map": {
					M: emptyMap,
				},
			},
			Key: map[string]*dynamodb.AttributeValue{
				"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(model.ChannelID, 10))},
				"Time":      &dynamodb.AttributeValue{S: aws.String(flatTime.Format(utils.DbTimeFormat))},
			},
			TableName:           aws.String(TableChannelUniquePlays),
			UpdateExpression:    aws.String("SET DeviceID = if_not_exists(DeviceID, :map)"),
			ConditionExpression: aws.String("attribute_not_exists(DeviceID)"),
		}

		query := update
		for {
			_, err := c.client.UpdateItemWithContext(ctx, query)
			if err != nil {
				if strings.Contains(err.Error(), PathError) {
					query = insert
				} else if strings.Contains(err.Error(), ConditionError) {
					query = update
				} else {
					return err
				}
			} else {
				return nil
			}
		}
	}
	return nil
}

// GetGeoMapGetPlatformMapsByTimesByTime return an array of PlatformMaps given
// channelID and startTime and endTime.
// Returns []models.PlatformMap if found and error
// if something went wrong.
func (c *videoPlayAdapter) GetPlatformMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.PlatformMap, error) {
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.PlatformMap{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableChannelPlatform),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		subResults := []models.PlatformMap{}
		err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &subResults)
		if err != nil {
			return nil, err
		}

		result = append(result, subResults...)

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetReferralMapsByTime return an array of ReferralMaps given
// channelID and startTime and endTime.
// Returns []models.ReferralMap if found and error
// if something went wrong.
func (c *videoPlayAdapter) GetReferralMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.ReferralMap, error) {
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.ReferralMap{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableChannelReferral),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		subResults := []models.ReferralMap{}
		err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &subResults)
		if err != nil {
			return nil, err
		}

		result = append(result, subResults...)

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetGeoMapsByTime return and array of GeoMap given
// channelID and startTime and endTime.
// Returns []models.GeoMap if found and error
// if something went wrong.
func (c *videoPlayAdapter) GetGeoMapsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.GeoMap, error) {
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.GeoMap{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableChannelGeo),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		subResults := []models.GeoMap{}
		err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &subResults)
		if err != nil {
			return nil, err
		}

		result = append(result, subResults...)

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

// GetUniqueViewsByTime return and array of UniqueViews given
// channelID and startTime and endTime.
// Returns []models.UniqueDevices if found and error
// if something went wrong.
func (c *videoPlayAdapter) GetUniqueViewsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]models.UniqueDevices, error) {
	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("ChannelID = :channelID AND #T BETWEEN :startTime AND :endTime")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			N: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(utils.DbTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(utils.DbTimeFormat)),
		},
	}
	attributePlaceholders := map[string]*string{
		"#T": aws.String("Time"),
	}

	result := []models.UniqueDevices{}
	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(TableChannelUniquePlays),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: conditionAttrValues,
			ExpressionAttributeNames:  attributePlaceholders,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, err
		}

		subResults := []models.UniqueDevices{}
		err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &subResults)
		if err != nil {
			return nil, err
		}

		result = append(result, subResults...)

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return result, nil
}

func getReferrerSourceTypeBucket(playObj *models.VideoPlay) string {
	contentLower := strings.ToLower(playObj.Content)
	mediumLower := strings.ToLower(playObj.Medium)
	playerLower := strings.ToLower(playObj.Player)

	if (playObj.HostChannelID > 0 ||
		playerLower == "frontpage" ||
		playerLower == "creative" ||
		len(contentLower) > 0 || len(mediumLower) > 0 ||
		(playObj.Referrer != nil && strings.Contains(strings.ToLower(playObj.Referrer.Host), "twitch"))) &&
		playerLower != "embed" {
		return "Internal"
	}

	return "External"
}

func getExternalReferrerSourceBucket(playObj *models.VideoPlay) string {
	playerLower := strings.ToLower(playObj.Player)

	if playerLower == "embed" {
		if playObj.ReferrerUrl == nil {
			return "unknown"
		}

		return parseExternalReferrerHostname(playObj.ReferrerUrl.Host)
	}

	if playObj.Referrer == nil {
		return "unknown"
	}

	return parseExternalReferrerHostname(playObj.Referrer.Host)
}

func parseExternalReferrerHostname(hostname string) string {
	if len(hostname) == 0 {
		return "unknown"
	}

	// Roll up all host names into one to save space in dynamo
	if strings.Contains(hostname, "google") {
		return "google"
	} else if strings.Contains(hostname, "facebook") {
		return "facebook"
	} else if strings.Contains(hostname, "reddit") {
		return "reddit"
	} else if strings.Contains(hostname, "youtube") {
		return "youtube"
	} else if strings.Contains(hostname, "twitter") || hostname == "t.co" {
		return "twitter"
	}

	return strings.ToLower(hostname)
}

func getInternalReferrerSourceBucket(playObj *models.VideoPlay) string {
	contentLower := strings.ToLower(playObj.Content)
	mediumLower := strings.ToLower(playObj.Medium)
	playerLower := strings.ToLower(playObj.Player)

	if playerLower == "frontpage" {
		return "front_page_featured"
	} else if mediumLower == "creative" {
		return "creative_page_featured"
	} else if playObj.HostChannelID > 0 {
		return "hosted"
	} else if mediumLower == "email" {
		return "email_live_notification"
	} else if mediumLower == "notification_toast" {
		return "onsite_notification"
	} else if mediumLower == "twitch_topnav" {
		if contentLower == "directory_following" {
			return "followed_channel"
		} else if contentLower == "directory_games" {
			return "directory_browse"
		}
		return "top_nav_bar"
	} else if mediumLower == "twitch_home" {
		if contentLower == "carousel" {
			return "front_page_featured"
		}
	} else if mediumLower == "twitch_following" {
		return "followed_channel"
	} else if mediumLower == "twitch_socialcolumn" {
		if contentLower == "recommended_channels" {
			return "recommended_channel"
		} else if contentLower == "followed_channels" {
			return "followed_channel"
		}
	} else if mediumLower == "twitch_directory" {
		if contentLower == "live_channel" {
			return "directory_browse"
		} else if contentLower == "language_stream" {
			return "directory_browse"
		}
	} else if mediumLower == "following_directory" {
		if contentLower == "live_channel" {
			return "followed_channel"
		}
	} else if mediumLower == "header_search" {
		return "search"
	} else if mediumLower == "clips_web" {
		return "clips_live"
	} else if mediumLower == "twitch_games_directory" {
		return "directory_browse"
	} else if mediumLower == "chat_bar_notification" {
		if contentLower == "host_channel" {
			return "hosted"
		}
	} else if contentLower == "friend_presence" {
		return "friend_presence"
	} else if mediumLower == "twitch_channel" {
		if contentLower == "raid_channel" {
			return "raid_channel"
		}
	}

	if playObj.Referrer.Host == "www.twitch.tv" {
		path := strings.Split(playObj.Referrer.Path, "/")

		if len(path) > 1 {
			primaryDir := strings.ToLower(path[1])

			// if the primary directory isnt registered in the ember router.js
			//	file we will assume this is a channel name
			if _, has := EmberPrimaryDirectories[primaryDir]; !has {
				if len(path[1]) == 0 {
					return "other"
				}

				return path[1]
			}

			// we will specifically handle www.twitch.tv/directory/... and
			//	www.twitch.tv/directory/following
			if primaryDir == "directory" {
				if len(path) > 2 && strings.ToLower(path[2]) == "following" {
					return "followed_channel"
				}
				return "directory_browse"
			}
		}
	}

	return "other"
}

func getPlatformBucket(platform string, player string) string {
	platformLower := strings.ToLower(platform)
	playerLower := strings.ToLower(player)

	if len(platformLower) == 0 {
		return "unknown"
	}

	if playerLower == "chromecast" ||
		playerLower == "roku" ||
		playerLower == "amazon" ||
		playerLower == "firetv" ||
		playerLower == "androidtv" ||
		playerLower == "firetv_stick" {
		return "tv"
	}

	if platformLower == "ios" {
		return "ios"
	} else if platformLower == "web" {
		return "web"
	} else if platformLower == "android" {
		return "android"
	} else if platformLower == "mobile_web" {
		return "mobile_web"
	} else if platformLower == "playstation" ||
		strings.HasPrefix(playerLower, "ps") ||
		strings.HasPrefix(platformLower, "ps") ||
		strings.HasPrefix(platformLower, "xbox") {
		return "console"
	}

	return "other"
}
