package app

import (
	"fmt"
	"time"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/kinesis_processor/app/processors"
	"code.justin.tv/cb/kinesis_processor/clients"
)

const (
	clipsVODType = "clip"
)

//
// KinesisProcessor main processing logic.
//
type KinesisProcessor interface {
	Process()
}

type kinesisProcessor struct {
	clients                        *clients.Clients
	channelUpdateProcessor         processors.ChannelUpdate
	channelConcurrentProcessor     processors.ChannelConcurrent
	broadcastMinuteProcessor       processors.MinuteBroadcast
	incrementalFollowProcessor     processors.IncrementalFollow
	followAggregateUpdateProcessor processors.FollowAggregateUpdate
	clipCreateProcessor            processors.ClipCreate
	clipVideoPlayProcessor         processors.ClipVideoPlay
	videoPlayProcessor             processors.VideoPlay
	chatProcessor                  processors.Chat
	commercialProcessor            processors.Commercial
	raidProcessor                  processors.Raid
}

// NewKinesisProcessor create instance of processor using clients.
func NewKinesisProcessor(clients *clients.Clients) KinesisProcessor {

	return &kinesisProcessor{
		clients:                        clients,
		channelUpdateProcessor:         processors.NewChannelUpdate(clients),
		channelConcurrentProcessor:     processors.NewChannelConcurrent(clients),
		broadcastMinuteProcessor:       processors.NewMinuteBroadcast(clients),
		incrementalFollowProcessor:     processors.NewIncrementalFollow(clients),
		followAggregateUpdateProcessor: processors.NewFollowAggregateUpdate(clients),
		clipCreateProcessor:            processors.NewClipCreate(clients),
		clipVideoPlayProcessor:         processors.NewClipVideoPlay(clients),
		videoPlayProcessor:             processors.NewVideoPlay(clients),
		chatProcessor:                  processors.NewChat(clients),
		commercialProcessor:            processors.NewCommercial(clients),
		raidProcessor:                  processors.NewRaid(clients),
	}
}

// Process run Firehouse Kinesis stream Get and populate DynamoDB
func (k *kinesisProcessor) Process() {

	defer func() {
		// After processing
		err := k.channelUpdateProcessor.After()
		if err != nil {
			log.WithError(err).Error("ChannelUpdateProcessor.After failed.")
		}
		err = k.broadcastMinuteProcessor.After()
		if err != nil {
			log.WithError(err).Error("BroadcastMinuteProdcessor.After failed.")
		}
		err = k.channelConcurrentProcessor.After()
		if err != nil {
			log.WithError(err).Error("ChannelConcurrentProcessor.After failed.")
		}
		err = k.incrementalFollowProcessor.After()
		if err != nil {
			log.WithError(err).Error("incrementalFollowProcessor.After failed.")
		}
		err = k.followAggregateUpdateProcessor.After()
		if err != nil {
			log.WithError(err).Error("followAggregateUpdateProcessor.After failed.")
		}
		err = k.clipCreateProcessor.After()
		if err != nil {
			log.WithError(err).Error("ClipCreateProcessor.After failed.")
		}
		err = k.clipVideoPlayProcessor.After()
		if err != nil {
			log.WithError(err).Error("ClipVideoPlayProcessor.After failed.")
		}
		err = k.videoPlayProcessor.After()
		if err != nil {
			log.WithError(err).Error("VideoPlayGeoProcessor.After failed.")
		}
		err = k.chatProcessor.After()
		if err != nil {
			log.WithError(err).Error("ChatProcessor.After failed.")
		}
		err = k.commercialProcessor.After()
		if err != nil {
			log.WithError(err).Error("CommercialProcessor.After failed.")
		}
		err = k.raidProcessor.After()
		if err != nil {
			log.WithError(err).Error("RaidProcessor.After failed.")
		}
	}()

	// Before processing file
	err := k.broadcastMinuteProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.channelUpdateProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.channelConcurrentProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.incrementalFollowProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.followAggregateUpdateProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.clipCreateProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.clipVideoPlayProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.videoPlayProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.chatProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.commercialProcessor.Before()
	if err != nil {
		panic(err)
	}

	err = k.raidProcessor.Before()
	if err != nil {
		panic(err)
	}

	for {
		result, err := k.clients.Firehose.Get()
		if err != nil {
			log.WithError(err).Error("Firehose.Get failed.")
			time.Sleep(30 * time.Second)
			continue
		}

		// No entry found
		if result.Key == "" {
			continue
		}

		fmt.Println(time.Now(), "Start processing ", result.Key)
		fmt.Println("Processing events. Total: ", len(result.Events))
		updatesCount := 0
		sessionsCount := 0
		concurrentCount := 0
		incrementalFollowCount := 0
		aggregateFollowCount := 0
		aggregateUnfollowCount := 0
		clipVideoPlayCount := 0
		clipCreateCount := 0
		videoPlayCount := 0
		chatCount := 0
		commercialCount := 0
		raidCount := 0

		// Channel all Events
		for _, event := range result.Events {
			err = nil

			switch event.Name {
			case "minute_broadcast":
				sessionsCount++
				err = k.broadcastMinuteProcessor.ProcessEvent(event)
				if err == processors.ErrInvalidMinuteBroadcastEvent {
					continue
				}
			case "channel_update":
				updatesCount++
				err = k.channelUpdateProcessor.ProcessEvent(event)
			case "channel_concurrents":
				concurrentCount++
				err = k.channelConcurrentProcessor.ProcessEvent(event)
				if err == processors.ErrInvalidChannelConcurrentEvent {
					continue
				}
			case "server_follow":
				incrementalFollowCount++
				err = k.incrementalFollowProcessor.ProcessEvent(event)
			case "follow":
				aggregateFollowCount++
				err = k.followAggregateUpdateProcessor.ProcessEvent(event)
			case "unfollow":
				aggregateUnfollowCount++
				err = k.followAggregateUpdateProcessor.ProcessEvent(event)
			case "video-play":
				videoPlayCount++
				err = k.videoPlayProcessor.ProcessEvent(event)

				if event.Fields["vod_type"] == clipsVODType {
					clipVideoPlayCount++
					err = k.clipVideoPlayProcessor.ProcessEvent(event)
				}
			case "create_clip":
				clipCreateCount++
				err = k.clipCreateProcessor.ProcessEvent(event)
			case "server_chat_message":
				chatCount++
				err = k.chatProcessor.ProcessEvent(event)
			case "commercial":
				commercialCount++
				err = k.commercialProcessor.ProcessEvent(event)
			case "raid_execute":
				raidCount++
				err = k.raidProcessor.ProcessEvent(event)
			}

			if err != nil {
				// Ignore duplicates error
				if processors.IsDuplicateError(err) {
					continue
				}

				log.WithError(err).WithFields(log.Fields{
					"event_name":   event.Name,
					"event_fields": event.Fields,
				}).Error("app: failed to process event")
				continue
			}
		}
		fmt.Println("channel_update:\t", updatesCount)
		fmt.Println("minute_broadcast:\t", sessionsCount)
		fmt.Println("channel_concurrents:\t", concurrentCount)
		fmt.Println("server_follow:\t", incrementalFollowCount)
		fmt.Println("follow:\t", aggregateFollowCount)
		fmt.Println("unfollow:\t", aggregateUnfollowCount)
		fmt.Println("clip_video_plays:\t", clipVideoPlayCount)
		fmt.Println("clip_creates:\t", clipCreateCount)
		fmt.Println("video_plays:\t", videoPlayCount)
		fmt.Println("server_chat_message:\t", chatCount)
		fmt.Println("commercials:\t", commercialCount)
		fmt.Println("raid:\t", raidCount)

		fmt.Println(time.Now(), "Finished processing ", result.Key)

		err = k.broadcastMinuteProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("BroadcastMinuteProcessor.Flush failed.")
			continue
		}

		err = k.channelConcurrentProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("ChannelConcurrentProcessor.Flush failed.")
			continue
		}

		err = k.channelUpdateProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("ChannelUpdateProcessor.Flush failed.")
			continue
		}

		err = k.incrementalFollowProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("incrementalFollowProcessor.Flush failed.")
			continue
		}

		err = k.followAggregateUpdateProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("followAggregateUpdateProcessor.Flush failed.")
			continue
		}

		err = k.videoPlayProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("videoPlayGeoProcessor.Flush failed.")
			continue
		}

		err = k.chatProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("chatProcessor.Flush failed.")
			continue
		}

		err = k.commercialProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("commercialProcessor.Flush failed.")
			continue
		}

		err = k.raidProcessor.Flush()
		if err != nil {
			log.WithError(err).Error("raidProcessor.Flush failed.")
			continue
		}

		err = k.clients.Firehose.DeleteObjectByKey(result.Key)
		if err != nil {
			log.WithError(err).Error("Firehose.Delete failed.")
			time.Sleep(30 * time.Second)
			continue
		}

		// Let things cooldown
		time.Sleep(5 * time.Second)
	}

}
