package processors

import (
	"context"
	"errors"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	ChannelConcurrentParallelWorkersCount = 500
	ChannelConcurrentBufferSize           = 25
)

// ErrInvalidChannelConcurrentEvent is the error returned
// when a "channel concurrent" event is malformed due to missing
// field(s) or invalid field type(s).
var ErrInvalidChannelConcurrentEvent = errors.New("processors: malformed channel concurrent event")

//
// ChannelConcurrent processor.
//
type ChannelConcurrent interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
	Flush() error
}

type channelConcurrent struct {
	clients *clients.Clients
	adapter adapters.ChannelConcurrentAdapter
	buffer  []models.ChannelConcurrent

	wg          sync.WaitGroup
	records     chan []models.ChannelConcurrent
	deadCounter int
}

// NewChannelConcurrent create new processor.
func NewChannelConcurrent(clients *clients.Clients) ChannelConcurrent {
	return &channelConcurrent{
		clients: clients,
		adapter: adapters.NewChannelConcurrentAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *channelConcurrent) Before() error {
	p.buffer = []models.ChannelConcurrent{}

	// Setup worker pool
	p.records = make(chan []models.ChannelConcurrent, ChannelConcurrentParallelWorkersCount)

	for w := 0; w < ChannelConcurrentParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *channelConcurrent) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute and retry
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()
		i := 0
		var err error
		for models := range p.records {
			i++
			if i%1 == 0 {
				//fmt.Println("U[",number,"] ",i, len(p.records))
			}

			err = p.adapter.BatchSave(context.TODO(), models)
			if err != nil {
				if IsDuplicateError(err) {
					continue
				}
				log.Error(err)
				continue
			}
		}
	}(number)
}

// ProcessEvent
func (p *channelConcurrent) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// Preparing fields for ChannelConcurrent model.
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return ErrInvalidChannelConcurrentEvent
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return err
		}
		eventTime = *result
	}

	// Preparing fields for ChannelConcurrent model.
	total, err := strconv.ParseInt(event.Fields["total"], 10, 64)
	if err != nil {
		return ErrInvalidChannelConcurrentEvent
	}

	// Parse time
	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	model := models.ChannelConcurrent{
		ChannelID:   channelID,
		Time:        eventTimeTime,
		Total:       total,
		ContentMode: event.Fields["content_mode"],
	}

	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= ChannelConcurrentBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *channelConcurrent) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}

	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush the buffer
func (p *channelConcurrent) Flush() error {
	// Copy buffer to temp thing
	temp := make([]models.ChannelConcurrent, len(p.buffer))
	copy(temp, p.buffer)
	p.records <- temp

	// Reset buffer
	p.buffer = []models.ChannelConcurrent{}
	return nil
}
