package processors

import (
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"

	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
	"runtime/debug"
)

const (
	ChannelSessionParallelWorkersCount = 50
)

//
// ChannelSession processor.
//
type ChannelSession interface {
	Before() error
	Process(minutes []models.MinuteBroadcast) error
	After() error
}

type channelSession struct {
	clients *clients.Clients
	adapter adapters.ChannelSessionAdapter

	wg      sync.WaitGroup
	records chan []models.MinuteBroadcast
}

// NewChannelSession create new processor.
func NewChannelSession(clients *clients.Clients) (ChannelSession, error) {
	adapter := adapters.NewChannelSessionAdapter(clients.Conf.Environment, clients.Conf.AWSRegion)

	return &channelSession{
		clients: clients,
		adapter: adapter,
	}, nil
}

// Before - start workers
func (p *channelSession) Before() error {

	// Setup worker pool
	p.records = make(chan []models.MinuteBroadcast, ChannelSessionParallelWorkersCount)

	for w := 0; w < ChannelSessionParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *channelSession) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Errorf("Failed worker: %s: %s", r, debug.Stack())
			}

			// Chill for a minute.
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		ignoreEventsBefore, err := time.Parse(utils.DbTimeFormat, "2017-09-15 00:00:00")
		if err != nil {
			log.Error(err)
			return
		}

		for record := range p.records {
			for _, model := range record {
				// Ignore events before time
				if ignoreEventsBefore.Sub(model.Time) >= 0 {
					continue
				}

				_, err := p.adapter.ExtendOrCreate(model)
				if err != nil {
					log.Error(err)
				}
			}

		}
	}(number)
}

// ProcessEvent
func (p *channelSession) Process(minutes []models.MinuteBroadcast) error {
	tmp := make([]models.MinuteBroadcast, len(minutes))
	copy(tmp, minutes)
	p.records <- minutes

	return nil
}

// After
func (p *channelSession) After() error {
	close(p.records)
	p.wg.Wait()
	return nil
}
