package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	ChannelUpdateParallelWorkersCount = 20
	ChannelUpdateBufferSize           = 25
)

//
// ChannelUpdate processor.
//
type ChannelUpdate interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
	Flush() error
}

type channelUpdate struct {
	clients *clients.Clients
	adapter adapters.ChannelUpdateAdapter
	buffer  []models.ChannelUpdate

	wg          sync.WaitGroup
	records     chan []models.ChannelUpdate
	deadCounter int
}

// NewChannelUpdate create new processor.
func NewChannelUpdate(clients *clients.Clients) ChannelUpdate {
	return &channelUpdate{
		clients: clients,
		adapter: adapters.NewChannelUpdateAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *channelUpdate) Before() error {
	p.buffer = []models.ChannelUpdate{}
	// Setup worker pool
	p.records = make(chan []models.ChannelUpdate, ChannelUpdateParallelWorkersCount)

	for w := 0; w < ChannelUpdateParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *channelUpdate) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()
		i := 0
		for models := range p.records {
			i++
			if i%1 == 0 {
				//fmt.Println("U[",number,"] ",i, len(p.records))
			}

			err := p.adapter.BatchSave(context.TODO(), models)
			if err != nil {
				if IsDuplicateError(err) {
					continue
				}
				log.Error(err)
				continue
			}

		}
	}(number)
}

// ProcessEvent
func (p *channelUpdate) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// Preparing fields for ChannelUpdate model.
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return err
	}
	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return err
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	model := models.ChannelUpdate{
		ChannelID:              channelID,
		Time:                   eventTimeTime,
		Game:                   event.Fields["game"],
		OldGame:                event.Fields["old_game"],
		GameID:                 event.Fields["game_id"],
		OldGameID:              event.Fields["old_game_id"],
		BroadcasterLanguage:    event.Fields["broadcaster_language"],
		OldBroadcasterLanguage: event.Fields["old_broadcaster_language"],
		Status:                 event.Fields["status"],
		OldStatus:              event.Fields["old_status"],
	}

	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= ChannelUpdateBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *channelUpdate) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush
func (p *channelUpdate) Flush() error {
	tmp := make([]models.ChannelUpdate, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.ChannelUpdate{}

	return nil
}
